/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.tests.file;

import io.vertx.core.Expectation;
import io.vertx.core.Vertx;
import io.vertx.core.VertxOptions;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.file.FileSystemOptions;
import io.vertx.core.file.impl.FileResolverImpl;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpResponseExpectation;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.impl.Utils;
import io.vertx.core.internal.VertxInternal;
import io.vertx.core.spi.file.FileResolver;
import io.vertx.test.core.VertxTestBase;
import io.vertx.test.http.HttpTestBase;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;

public abstract class FileResolverTestBase
extends VertxTestBase {
    private final String cacheBaseDir;
    protected FileResolverImpl resolver;
    private ClassLoader testCL;

    public FileResolverTestBase() {
        try {
            this.cacheBaseDir = new File(System.getProperty("java.io.tmpdir", ".") + File.separator + "vertx-cache").getCanonicalPath();
        }
        catch (IOException e) {
            throw new IllegalStateException("Cannot resolve the canonical path to the cache dir", e);
        }
    }

    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.testCL = Thread.currentThread().getContextClassLoader();
        File baseDir = new File(new File(new File(new File("src"), "test"), "classpath"), "fileresolver");
        this.assertTrue(baseDir.exists() && baseDir.isDirectory());
        ClassLoader resourcesLoader = this.resourcesLoader(baseDir);
        Thread.currentThread().setContextClassLoader(resourcesLoader);
        this.resolver = new FileResolverImpl();
    }

    @Override
    protected void tearDown() throws Exception {
        Thread.currentThread().setContextClassLoader(this.testCL);
        this.resolver.close();
        super.tearDown();
    }

    protected ClassLoader resourcesLoader(File baseDir) throws Exception {
        return Thread.currentThread().getContextClassLoader();
    }

    private static File resolveFile(VertxInternal vertx, String na) {
        return vertx.fileResolver().resolve(na);
    }

    @Test
    public void testReadFileInDirThenReadDir() {
        Buffer buff = this.vertx.fileSystem().readFileBlocking("webroot/subdir/subfile.html");
        this.assertEquals(buff.toString(), "<html><body>subfile</body></html>");
        Set names = this.vertx.fileSystem().readDirBlocking("webroot/subdir").stream().map(path -> {
            int idx = path.lastIndexOf(File.separator);
            return idx == -1 ? path : path.substring(idx + 1);
        }).collect(Collectors.toSet());
        this.assertEquals(names, new HashSet<String>(Arrays.asList("subdir2", "subfile.html")));
    }

    @Test
    public void testResolveNotExistFile() {
        File file = this.resolver.resolve("doesnotexist.txt");
        this.assertFalse(file.exists());
        this.assertEquals("doesnotexist.txt", file.getPath());
    }

    @Test
    public void testResolveNotExistDirectory() {
        File file = this.resolver.resolve("somedir");
        this.assertFalse(file.exists());
        this.assertEquals("somedir", file.getPath());
    }

    @Test
    public void testResolveNotExistFileInDirectory() {
        File file = this.resolver.resolve("somedir/doesnotexist.txt");
        this.assertFalse(file.exists());
        this.assertEquals("somedir" + File.separator + "doesnotexist.txt", file.getPath());
    }

    @Test
    public void testResolveFileFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("afile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>afile</body></html>", this.readFile(file));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testResolveFileFromClasspathDisableCaching() throws Exception {
        try (VertxInternal vertx = (VertxInternal)Vertx.vertx((VertxOptions)new VertxOptions().setFileSystemOptions(new FileSystemOptions().setFileCachingEnabled(true)));){
            for (int i = 0; i < 2; ++i) {
                File file = FileResolverTestBase.resolveFile(vertx, "afile.html");
                this.assertTrue(file.exists());
                this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
                this.assertFalse(file.isDirectory());
                this.assertEquals("<html><body>afile</body></html>", this.readFile(file));
            }
        }
    }

    @Test
    public void testResolveFileWithSpacesFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("afile with spaces.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>afile with spaces</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testResolveFileWithSpaceAtEndFromClasspath() {
        Assume.assumeFalse((boolean)Utils.isWindows());
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("afilewithspaceatend ");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("afilewithspaceatend ", this.readFile(file));
        }
    }

    @Test
    public void testCacheDirIsPosix0700() throws Exception {
        Assume.assumeFalse((boolean)Utils.isWindows());
        File file = this.resolver.resolve("afile with spaces.html");
        this.assertTrue(file.exists());
        this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
        File parent = file.getParentFile();
        this.assertEquals("rwx------", PosixFilePermissions.toString(Files.getPosixFilePermissions(parent.toPath(), new LinkOption[0])));
    }

    @Test
    public void testResolveDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("webroot");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertTrue(file.isDirectory());
        }
    }

    @Test
    public void testResolveFileInDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("webroot/somefile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>blah</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testResolveSubDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("webroot/subdir");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertTrue(file.isDirectory());
        }
    }

    @Test
    public void testResolveFileInSubDirectoryFromClasspath() throws Exception {
        for (int i = 0; i < 2; ++i) {
            File file = this.resolver.resolve("webroot/subdir/subfile.html");
            this.assertTrue(file.exists());
            this.assertTrue(file.getPath().startsWith(this.cacheBaseDir + "-"));
            this.assertFalse(file.isDirectory());
            this.assertEquals("<html><body>subfile</body></html>", this.readFile(file));
        }
    }

    @Test
    public void testRecursivelyUnpack() throws Exception {
        File file = this.resolver.resolve("webroot/subdir");
        this.assertTrue(file.exists());
        File sub = new File(file, "subfile.html");
        this.assertTrue(sub.exists());
        this.assertEquals("<html><body>subfile</body></html>", this.readFile(sub));
    }

    @Test
    public void testRecursivelyUnpack2() throws Exception {
        File file = this.resolver.resolve("webroot/subdir");
        this.assertTrue(file.exists());
        File sub = new File(new File(file, "subdir2"), "subfile2.html");
        this.assertTrue(sub.exists());
        this.assertEquals("<html><body>subfile2</body></html>", this.readFile(sub));
    }

    @Test
    public void testDeleteCacheDir() throws Exception {
        FileResolverImpl resolver2 = new FileResolverImpl();
        File file = resolver2.resolve("webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        resolver2.close();
        this.assertFalse(cacheDir.exists());
    }

    @Test
    public void testCacheDirDeletedOnVertxClose() {
        VertxInternal vertx2 = (VertxInternal)this.vertx();
        File file = FileResolverTestBase.resolveFile(vertx2, "webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        vertx2.close().onComplete(this.onSuccess(v -> {
            this.assertFalse(cacheDir.exists());
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testFileSystemReadFile() {
        this.assertTrue(this.vertx.fileSystem().existsBlocking("afile.html"));
        this.assertFalse(this.vertx.fileSystem().propsBlocking("afile.html").isDirectory());
        Buffer buffer = this.vertx.fileSystem().readFileBlocking("afile.html");
        this.assertNotNull(buffer);
        this.assertTrue(buffer.toString().startsWith("<html><body>afile</body></html>"));
    }

    @Test
    public void testFileSystemReadDirectory() {
        this.assertTrue(this.vertx.fileSystem().existsBlocking("webroot"));
        this.assertTrue(this.vertx.fileSystem().propsBlocking("webroot").isDirectory());
    }

    @Test
    public void testSendFileFromClasspath() {
        this.waitFor(2);
        this.vertx.createHttpServer(new HttpServerOptions().setPort(HttpTestBase.DEFAULT_HTTP_PORT)).requestHandler(res -> res.response().sendFile("webroot/somefile.html").onComplete(this.onSuccess(v -> this.complete()))).listen().onComplete(this.onSuccess(res -> this.vertx.createHttpClient(new HttpClientOptions()).request(HttpMethod.GET, HttpTestBase.DEFAULT_HTTP_PORT, "localhost", "/").compose(req -> req.send().expecting((Expectation)HttpResponseExpectation.SC_OK).compose(HttpClientResponse::body)).onComplete(this.onSuccess(body -> {
            this.assertTrue(body.toString().startsWith("<html><body>blah</body></html>"));
            this.complete();
        }))));
        this.await();
    }

    @Test
    public void testResolveFileFromDifferentThreads() throws Exception {
        int size = 0xA00000;
        byte[] content = new byte[size];
        new Random().nextBytes(content);
        File out = new File("target/test-classes/temp");
        if (out.exists()) {
            Files.delete(out.toPath());
        }
        Files.write(out.toPath(), content, StandardOpenOption.CREATE_NEW);
        int count = 100;
        CountDownLatch latch = new CountDownLatch(count);
        CountDownLatch start = new CountDownLatch(1);
        ArrayList errors = new ArrayList();
        for (int i = 0; i < count; ++i) {
            Runnable runnable = () -> {
                try {
                    start.await();
                    File file = this.resolver.resolve("temp");
                    byte[] data = Files.readAllBytes(file.toPath());
                    Assert.assertArrayEquals((byte[])content, (byte[])data);
                }
                catch (Exception e) {
                    errors.add(e);
                }
                finally {
                    latch.countDown();
                }
            };
            new Thread(runnable).start();
        }
        start.countDown();
        latch.await();
        this.assertTrue(errors.isEmpty());
    }

    @Test
    public void testEnableCaching() throws Exception {
        this.testCaching(true);
    }

    @Test
    public void testDisableCaching() throws Exception {
        this.testCaching(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testCaching(boolean enabled) throws Exception {
        VertxInternal vertx = (VertxInternal)Vertx.vertx((VertxOptions)new VertxOptions().setFileSystemOptions(new FileSystemOptions().setFileCachingEnabled(enabled)));
        File tmp = File.createTempFile("vertx", ".bin");
        tmp.deleteOnExit();
        final URL url = tmp.toURI().toURL();
        Files.write(tmp.toPath(), "foo".getBytes(), new OpenOption[0]);
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(new ClassLoader(){

                @Override
                public URL getResource(String name) {
                    if ("foo".equals(name)) {
                        return url;
                    }
                    return super.getResource(name);
                }
            });
            File f = FileResolverTestBase.resolveFile(vertx, "foo");
            this.assertEquals("foo", new String(Files.readAllBytes(f.toPath())));
            Files.write(tmp.toPath(), "bar".getBytes(), new OpenOption[0]);
            f = FileResolverTestBase.resolveFile(vertx, "foo");
            if (enabled) {
                this.assertEquals("foo", new String(Files.readAllBytes(f.toPath())));
            } else {
                this.assertEquals("bar", new String(Files.readAllBytes(f.toPath())));
            }
        }
        finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    private String readFile(File file) {
        return this.vertx.fileSystem().readFileBlocking(file.getAbsolutePath()).toString();
    }

    @Test
    public void testResolveAfterCloseThrowsISE() throws Exception {
        FileResolverImpl resolver2 = new FileResolverImpl();
        File file = resolver2.resolve("webroot/somefile.html");
        this.assertTrue(file.exists());
        File cacheDir = file.getParentFile().getParentFile();
        this.assertTrue(cacheDir.exists());
        resolver2.close();
        this.assertFalse(cacheDir.exists());
        try {
            resolver2.resolve("webroot/somefile.html");
            this.fail("Should fail");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    @Test
    public void testResolveRelativeFileInDirectoryFromClasspath() {
        File somefile = this.resolver.resolve("a/a.txt");
        this.assertTrue(somefile.exists());
        this.assertTrue(somefile.getPath().startsWith(this.cacheBaseDir + "-"));
        this.assertFalse(somefile.isDirectory());
        File someotherfile = new File(new File(somefile.getParentFile().getParentFile(), "b"), "b.txt");
        this.assertFalse(someotherfile.exists());
        someotherfile = this.resolver.resolve(someotherfile.getAbsolutePath());
        this.assertTrue(someotherfile.exists());
        this.assertTrue(someotherfile.getPath().startsWith(this.cacheBaseDir + "-"));
        this.assertFalse(someotherfile.isDirectory());
    }

    @Test
    public void testDoNotResolveAbsoluteFileInDirectoryFromClasspath() {
        File somefile = this.resolver.resolve("/a/a.txt");
        this.assertFalse(somefile.exists());
    }

    @Test
    public void testResolveCacheDir() {
        File somefile = this.resolver.resolve("");
        this.assertTrue(somefile.exists());
        this.assertTrue(somefile.getPath().startsWith(this.cacheBaseDir + "-"));
        this.assertTrue(somefile.isDirectory());
    }

    @Test
    public void testReadFileInDirThenReadDirMultipleLevels() {
        Buffer buff = this.vertx.fileSystem().readFileBlocking("tree/a/b/c.txt");
        this.assertNotNull(buff);
        Function<List, Set> fx = l -> l.stream().map(path -> {
            int idx = path.lastIndexOf(File.separator);
            return idx == -1 ? path : path.substring(idx + 1);
        }).collect(Collectors.toSet());
        Set names = fx.apply(this.vertx.fileSystem().readDirBlocking("tree/a/b"));
        this.assertEquals(new HashSet<String>(Arrays.asList("c.txt")), names);
        names = fx.apply(this.vertx.fileSystem().readDirBlocking("tree/a"));
        this.assertEquals(new HashSet<String>(Arrays.asList("b", "b.txt")), names);
    }

    @Test
    public void testBugEndWithSlash() {
        FileResolver resolver = ((VertxInternal)this.vertx).fileResolver();
        File f = resolver.resolve("tree/");
        this.assertNotNull(f);
        this.assertTrue(f.isDirectory());
        this.assertEquals("tree", f.getName());
    }

    @Test
    public void testReadFileInDirThenReadDirMultipleLevelsMissingResource() {
        Buffer buff = this.vertx.fileSystem().readFileBlocking("tree/a/b/c.txt");
        this.assertNotNull(buff);
        Function<List, Set> fx = l -> l.stream().map(path -> {
            int idx = path.lastIndexOf(File.separator);
            return idx == -1 ? path : path.substring(idx + 1);
        }).collect(Collectors.toSet());
        try {
            this.vertx.fileSystem().readDirBlocking("tree/a/d");
            this.fail("should fail as it doesn't exist");
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
        try {
            this.vertx.fileSystem().readDirBlocking("tree/a/d");
            this.fail("should fail as it doesn't exist");
        }
        catch (RuntimeException runtimeException) {
            // empty catch block
        }
    }

    @Test
    public void testGetTheCacheDirWithoutHacks() {
        String cacheDir = this.resolver.cacheDir();
        if (cacheDir != null) {
            this.assertTrue(cacheDir.startsWith(this.cacheBaseDir + "-"));
            String uuid = cacheDir.substring(this.cacheBaseDir.length() + 1);
            try {
                UUID.fromString(uuid);
            }
            catch (Exception e) {
                this.fail("Expected a UUID");
            }
        }
    }
}

