/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.tests.net;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.NetClient;
import io.vertx.core.net.NetServer;
import io.vertx.core.net.NetServerOptions;
import io.vertx.core.net.SocketAddress;
import io.vertx.core.net.TrafficShapingOptions;
import io.vertx.core.transport.Transport;
import io.vertx.test.core.TestUtils;
import io.vertx.test.core.VertxTestBase;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class NetBandwidthLimitingTest
extends VertxTestBase {
    public static final String DEFAULT_HOST = "localhost";
    private static final int OUTBOUND_LIMIT = 65536;
    private static final int INBOUND_LIMIT = 65536;
    @Rule
    public TemporaryFolder testFolder = new TemporaryFolder();
    private SocketAddress testAddress;
    private NetClient client = null;
    private final List<NetServer> servers = Collections.synchronizedList(new ArrayList());

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        if (USE_DOMAIN_SOCKETS) {
            this.assertTrue("Native transport not enabled", TRANSPORT.implementation().supportsDomainSockets());
            File tmp = TestUtils.tmpFile(".sock");
            this.testAddress = SocketAddress.domainSocketAddress((String)tmp.getAbsolutePath());
        } else {
            this.testAddress = SocketAddress.inetSocketAddress((int)1234, (String)DEFAULT_HOST);
        }
        this.client = this.vertx.createNetClient();
    }

    @Override
    protected void tearDown() throws Exception {
        if (this.client != null) {
            try {
                this.client.close().await();
            }
            finally {
                this.client = null;
            }
        }
        while (!this.servers.isEmpty()) {
            Iterator<NetServer> it = this.servers.iterator();
            NetServer server = it.next();
            it.remove();
            server.close().await();
        }
        super.tearDown();
    }

    @Test
    public void sendBufferThrottled() {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> sock.handler(buf -> sock.write((Object)expected)));
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> {
                sock.handler(buff -> {
                    received.appendBuffer(buff);
                    if (received.length() == expected.length()) {
                        long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                        this.assertEquals(expected, received);
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                        this.testComplete();
                    }
                });
                sock.write("foo");
            }));
        }));
        this.await();
    }

    @Test
    public void sendFileIsThrottled() throws Exception {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        File fDir = this.testFolder.newFolder();
        String content = TestUtils.randomUnicodeString(60000);
        File file = this.setupFile(fDir.toString(), "some-file.txt", content);
        Buffer expected = Buffer.buffer((String)content);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> sock.handler(buf -> sock.sendFile(file.getAbsolutePath())));
        Future result = server.listen(this.testAddress);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> {
                sock.handler(buff -> {
                    received.appendBuffer(buff);
                    if (received.length() == expected.length()) {
                        long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                        this.assertEquals(expected, received);
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                        this.testComplete();
                    }
                });
                sock.write("foo");
            }));
        }));
        this.await();
    }

    @Test
    public void dataUploadIsThrottled() {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> {
            sock.handler(buff -> {
                received.appendBuffer(buff);
                if (received.length() == expected.length()) {
                    long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                    this.assertEquals(expected, received);
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                    this.testComplete();
                }
            });
            sock.write("foo");
        });
        server.listen(this.testAddress).await();
        Future clientConnect = this.client.connect(this.testAddress);
        clientConnect.onComplete(this.onSuccess(sock -> sock.handler(buf -> sock.write((Object)expected))));
        this.await();
    }

    @Test
    public void fileUploadIsThrottled() throws Exception {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        File fDir = this.testFolder.newFolder();
        String content = TestUtils.randomUnicodeString(60000);
        File file = this.setupFile(fDir.toString(), "some-file.txt", content);
        Buffer expected = Buffer.buffer((String)content);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> {
            sock.handler(buff -> {
                received.appendBuffer(buff);
                if (received.length() == expected.length()) {
                    long expectedTimeInMillis = this.expectedTimeMillis(received.length(), 65536);
                    this.assertEquals(expected, received);
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
                    this.testComplete();
                }
            });
            sock.write("foo");
        });
        server.listen(this.testAddress).await();
        Future clientConnect = this.client.connect(this.testAddress);
        clientConnect.onComplete(this.onSuccess(sock -> sock.handler(buf -> sock.sendFile(file.getAbsolutePath()))));
        this.await();
    }

    @Test
    public void testSendBufferIsTrafficShapedWithSharedServers() throws Exception {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        final Buffer expected = TestUtils.randomBuffer(262144);
        int numEventLoops = 4;
        Future listenLatch = this.vertx.deployVerticle(() -> new AbstractVerticle(){

            public void start(Promise<Void> startPromise) {
                NetServer testServer = NetBandwidthLimitingTest.this.netServer();
                testServer.connectHandler(sock -> sock.handler(buf -> sock.write((Object)expected)));
                testServer.listen(NetBandwidthLimitingTest.this.testAddress).onComplete(v -> startPromise.complete());
            }
        }, new DeploymentOptions().setInstances(numEventLoops));
        AtomicLong startTime = new AtomicLong();
        CountDownLatch waitForResponse = new CountDownLatch(4);
        listenLatch.onComplete(this.onSuccess(v -> {
            startTime.set(System.nanoTime());
            for (int i = 0; i < 4; ++i) {
                Buffer received = Buffer.buffer();
                Future clientConnect = this.client.connect(this.testAddress);
                clientConnect.onComplete(this.onSuccess(sock -> {
                    sock.handler(buff -> {
                        received.appendBuffer(buff);
                        if (received.length() == expected.length()) {
                            this.assertEquals(expected, received);
                            waitForResponse.countDown();
                        }
                    });
                    sock.write("foo");
                }));
            }
        }));
        waitForResponse.await();
        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime.get());
        long expectedTimeInMillis = this.expectedTimeMillis(expected.length() * 4, 65536);
        this.assertTimeTakenFallsInRange(expectedTimeInMillis, elapsedMillis);
    }

    @Test
    public void testDynamicInboundRateUpdate() {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> {
            sock.handler(buff -> {
                received.appendBuffer(buff);
                if (received.length() == expected.length()) {
                    this.assertEquals(expected, received);
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                    this.assertTrue(elapsedMillis < this.expectedUpperBoundTimeMillis(received.length(), 65536));
                    this.testComplete();
                }
            });
            sock.write("foo");
        });
        server.listen(this.testAddress).await();
        TrafficShapingOptions trafficOptions = new TrafficShapingOptions().setOutboundGlobalBandwidth(65536L).setInboundGlobalBandwidth(131072L);
        server.updateTrafficShapingOptions(trafficOptions);
        Future clientConnect = this.client.connect(this.testAddress);
        clientConnect.onComplete(this.onSuccess(sock -> sock.handler(buf -> sock.write((Object)expected))));
        this.await();
    }

    @Test
    public void testDynamicOutboundRateUpdate() {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        long startTime = System.nanoTime();
        Buffer expected = TestUtils.randomBuffer(262144);
        Buffer received = Buffer.buffer();
        NetServer server = this.netServer();
        server.connectHandler(sock -> sock.handler(buf -> sock.write((Object)expected)));
        Future result = server.listen(this.testAddress);
        TrafficShapingOptions trafficOptions = new TrafficShapingOptions().setInboundGlobalBandwidth(65536L).setOutboundGlobalBandwidth(131072L);
        server.updateTrafficShapingOptions(trafficOptions);
        result.onComplete(this.onSuccess(resp -> {
            Future clientConnect = this.client.connect(this.testAddress);
            clientConnect.onComplete(this.onSuccess(sock -> {
                sock.handler(buff -> {
                    received.appendBuffer(buff);
                    if (received.length() == expected.length()) {
                        this.assertEquals(expected, received);
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                        this.assertTrue(elapsedMillis < this.expectedUpperBoundTimeMillis(received.length(), 65536));
                        this.testComplete();
                    }
                });
                sock.write("foo");
            }));
        }));
        this.await();
    }

    @Test(expected=IllegalStateException.class)
    public void testRateUpdateWhenServerStartedWithoutTrafficShaping() throws Exception {
        Assume.assumeFalse((TRANSPORT == Transport.IO_URING ? 1 : 0) != 0);
        NetServerOptions options = new NetServerOptions().setHost(DEFAULT_HOST).setPort(0);
        NetServer testServer = this.netServer(options);
        testServer.listen().await(20L, TimeUnit.SECONDS);
        TrafficShapingOptions trafficOptions = new TrafficShapingOptions().setOutboundGlobalBandwidth(65536L).setInboundGlobalBandwidth(131072L);
        testServer.updateTrafficShapingOptions(trafficOptions).await(20L, TimeUnit.SECONDS);
    }

    private long expectedTimeMillis(int size, int rate) {
        return TimeUnit.MILLISECONDS.convert(size / rate, TimeUnit.SECONDS);
    }

    private long expectedUpperBoundTimeMillis(int size, int rate) {
        return TimeUnit.MILLISECONDS.convert(size / rate, TimeUnit.SECONDS);
    }

    private void assertTimeTakenFallsInRange(long expectedTimeInMillis, long actualTimeInMillis) {
        Assert.assertTrue((actualTimeInMillis >= expectedTimeInMillis - 2000L ? 1 : 0) != 0);
        Assert.assertTrue((actualTimeInMillis <= expectedTimeInMillis + 2000L ? 1 : 0) != 0);
    }

    private NetServer netServer() {
        return this.netServer(new NetServerOptions().setHost(DEFAULT_HOST).setPort(0).setTrafficShapingOptions(new TrafficShapingOptions().setInboundGlobalBandwidth(65536L).setOutboundGlobalBandwidth(65536L)));
    }

    private NetServer netServer(NetServerOptions options) {
        NetServer server = this.vertx.createNetServer(options);
        this.servers.add(server);
        return server;
    }

    private File setupFile(String testDir, String fileName, String content) throws Exception {
        File file = new File(testDir, fileName);
        if (file.exists()) {
            file.delete();
        }
        BufferedWriter out = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file), "UTF-8"));
        out.write(content);
        out.close();
        return file;
    }
}

