/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.tests.parsetools;

import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.DecodeException;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.parsetools.JsonEvent;
import io.vertx.core.parsetools.JsonEventType;
import io.vertx.core.parsetools.JsonParser;
import io.vertx.core.streams.ReadStream;
import io.vertx.test.core.TestUtils;
import io.vertx.tests.parsetools.FakeStream;
import java.math.BigInteger;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import junit.framework.TestCase;
import org.junit.Assert;
import org.junit.Test;

public class JsonParserTest {
    @Test
    public void testParseEmptyObject() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.handler(event -> {
            Assert.assertNull((Object)event.fieldName());
            Assert.assertNull((Object)event.value());
            switch (status.getAndIncrement()) {
                case 0: {
                    Assert.assertEquals((Object)JsonEventType.START_OBJECT, (Object)event.type());
                    break;
                }
                case 1: {
                    Assert.assertEquals((Object)JsonEventType.END_OBJECT, (Object)event.type());
                    break;
                }
                default: {
                    Assert.fail();
                }
            }
        });
        AtomicInteger count = new AtomicInteger();
        parser.endHandler(v -> count.incrementAndGet());
        parser.handle((Object)Buffer.buffer((String)"{}"));
        Assert.assertEquals((long)2L, (long)status.get());
        Assert.assertEquals((long)0L, (long)count.get());
        parser.end();
        Assert.assertEquals((long)1L, (long)count.get());
        Assert.assertEquals((long)2L, (long)status.get());
        try {
            parser.end();
            Assert.fail();
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    @Test
    public void testParseEmptyArray() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.handler(event -> {
            Assert.assertNull((Object)event.fieldName());
            Assert.assertNull((Object)event.value());
            switch (status.getAndIncrement()) {
                case 0: {
                    Assert.assertEquals((Object)JsonEventType.START_ARRAY, (Object)event.type());
                    break;
                }
                case 1: {
                    Assert.assertEquals((Object)JsonEventType.END_ARRAY, (Object)event.type());
                    break;
                }
                default: {
                    Assert.fail();
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"[]"));
        Assert.assertEquals((long)2L, (long)status.get());
    }

    @Test
    public void parseUnfinishedThrowingException() {
        StringBuilder events = new StringBuilder();
        JsonParser parser = JsonParser.newParser();
        parser.handler(e -> events.append("json,"));
        parser.endHandler(v -> events.append("end,"));
        parser.handle((Object)Buffer.buffer((String)"{\"un\":\"finished\""));
        try {
            parser.end();
            Assert.fail();
        }
        catch (DecodeException decodeException) {
            // empty catch block
        }
        Assert.assertEquals((Object)"json,json,", (Object)events.toString());
    }

    @Test
    public void parseUnfinishedExceptionHandler() {
        StringBuilder events = new StringBuilder();
        JsonParser parser = JsonParser.newParser();
        parser.handler(e -> events.append("json,"));
        parser.endHandler(v -> events.append("end,"));
        parser.exceptionHandler(e -> events.append("exception,"));
        parser.handle((Object)Buffer.buffer((String)"{\"un\":\"finished\""));
        parser.end();
        Assert.assertEquals((Object)"json,json,exception,end,", (Object)events.toString());
    }

    @Test
    public void testParseWithErrors() {
        Buffer data = Buffer.buffer((String)"{\"foo\":\"foo_value\"},{\"bar\":\"bar_value\"},{\"juu\":\"juu_value\"}");
        JsonParser parser = JsonParser.newParser();
        ArrayList objects = new ArrayList();
        ArrayList errors = new ArrayList();
        AtomicInteger endCount = new AtomicInteger();
        parser.objectValueMode().handler(event -> objects.add(event.objectValue())).exceptionHandler(errors::add).endHandler(v -> endCount.incrementAndGet());
        parser.write(data);
        Assert.assertEquals((long)3L, (long)objects.size());
        List<JsonObject> expected = Arrays.asList(new JsonObject().put("foo", (Object)"foo_value"), new JsonObject().put("bar", (Object)"bar_value"), new JsonObject().put("juu", (Object)"juu_value"));
        Assert.assertEquals(expected, objects);
        Assert.assertEquals((long)2L, (long)errors.size());
        Assert.assertEquals((long)0L, (long)endCount.get());
        objects.clear();
        errors.clear();
        parser.end();
        Assert.assertEquals(Collections.emptyList(), objects);
        Assert.assertEquals(Collections.emptyList(), errors);
        Assert.assertEquals((long)1L, (long)endCount.get());
    }

    @Test
    public void testParseObjectValue() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.objectValueMode();
        JsonObject expected = new JsonObject().put("number", (Object)3).put("floating", (Object)3.5).put("true", (Object)true).put("false", (Object)false).put("string", (Object)"s").put("object", (Object)new JsonObject().put("foo", (Object)"bar")).put("array", (Object)new JsonArray().add((Object)0).add((Object)1).add((Object)2)).putNull("null").put("bytes", (Object)new byte[]{1, 2, 3});
        parser.handler(event -> {
            Assert.assertEquals((long)0L, (long)status.getAndIncrement());
            Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
            Assert.assertEquals((Object)expected, (Object)event.value());
        });
        parser.handle((Object)expected.toBuffer());
        Assert.assertEquals((long)1L, (long)status.get());
    }

    @Test
    public void testParseArrayValue() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.arrayValueMode();
        JsonArray expected = new JsonArray().add((Object)3).add((Object)3.5).add((Object)true).add((Object)false).add((Object)"s").addNull().add((Object)new JsonObject().put("foo", (Object)"bar")).add((Object)new JsonArray().add((Object)0).add((Object)1).add((Object)2)).add((Object)new byte[]{1, 2, 3});
        parser.handler(event -> {
            Assert.assertEquals((Object)expected, (Object)event.value());
            Assert.assertEquals((long)0L, (long)status.getAndIncrement());
        });
        parser.handle((Object)expected.toBuffer());
        Assert.assertEquals((long)1L, (long)status.get());
    }

    private void assertThrowCCE(JsonEvent event, Consumer<JsonEvent> ... checks) {
        for (Consumer<JsonEvent> check : checks) {
            try {
                check.accept(event);
                Assert.fail();
            }
            catch (ClassCastException classCastException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testStringValue() {
        this.testValue("\"bar\"", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)"bar", (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            Assert.assertTrue((boolean)event.isString());
            Assert.assertEquals((Object)"bar", (Object)event.stringValue());
            this.assertThrowCCE((JsonEvent)event, JsonEvent::integerValue, JsonEvent::longValue, JsonEvent::floatValue, JsonEvent::doubleValue, JsonEvent::booleanValue, JsonEvent::objectValue, JsonEvent::arrayValue);
            try {
                event.instantValue();
                Assert.fail();
            }
            catch (DateTimeParseException dateTimeParseException) {
                // empty catch block
            }
        }));
    }

    @Test
    public void testInstantValue() {
        Instant value = Instant.now();
        String encoded = DateTimeFormatter.ISO_INSTANT.format(value);
        this.testValue("\"" + encoded + "\"", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)encoded, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            Assert.assertTrue((boolean)event.isString());
            Assert.assertEquals((Object)encoded, (Object)event.stringValue());
            Assert.assertEquals((Object)value, (Object)event.instantValue());
            this.assertThrowCCE((JsonEvent)event, JsonEvent::integerValue, JsonEvent::longValue, JsonEvent::floatValue, JsonEvent::doubleValue, JsonEvent::booleanValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
    }

    @Test
    public void testBinaryValue() {
        byte[] value = TestUtils.randomByteArray(10);
        String encoded = Base64.getUrlEncoder().withoutPadding().encodeToString(value);
        this.testValue("\"" + encoded + "\"", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)encoded, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            Assert.assertTrue((boolean)event.isString());
            Assert.assertEquals((Object)encoded, (Object)event.stringValue());
            Assert.assertEquals((Object)Buffer.buffer((byte[])value), (Object)event.binaryValue());
            this.assertThrowCCE((JsonEvent)event, JsonEvent::integerValue, JsonEvent::longValue, JsonEvent::floatValue, JsonEvent::doubleValue, JsonEvent::booleanValue, JsonEvent::objectValue, JsonEvent::arrayValue);
            try {
                event.instantValue();
                Assert.fail();
            }
            catch (DateTimeParseException dateTimeParseException) {
                // empty catch block
            }
        }));
    }

    @Test
    public void testNullValue() {
        this.testValue("null", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals(null, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            Assert.assertTrue((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            Assert.assertNull((Object)event.integerValue());
            Assert.assertNull((Object)event.longValue());
            Assert.assertNull((Object)event.floatValue());
            Assert.assertNull((Object)event.doubleValue());
            Assert.assertNull((Object)event.binaryValue());
            Assert.assertNull((Object)event.instantValue());
            Assert.assertNull((Object)event.objectValue());
            Assert.assertNull((Object)event.arrayValue());
            Assert.assertNull((Object)event.stringValue());
            Assert.assertNull((Object)event.binaryValue());
        }));
    }

    @Test
    public void testLongValue() {
        this.testValue("567", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)567L, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            Assert.assertTrue((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            Assert.assertEquals((long)567L, (long)event.integerValue().intValue());
            Assert.assertEquals((long)567L, (long)event.longValue());
            Assert.assertEquals((float)567.0f, (float)event.floatValue().floatValue(), (float)0.01f);
            Assert.assertEquals((double)567.0, (double)event.doubleValue(), (double)0.01);
            this.assertThrowCCE((JsonEvent)event, JsonEvent::stringValue, JsonEvent::booleanValue, JsonEvent::binaryValue, JsonEvent::instantValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
    }

    @Test
    public void testDoubleValue() {
        this.testValue("567.45", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)567.45, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            Assert.assertTrue((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            Assert.assertEquals((long)567L, (long)event.integerValue().intValue());
            Assert.assertEquals((long)567L, (long)event.longValue());
            Assert.assertEquals((float)567.45f, (float)event.floatValue().floatValue(), (float)0.01f);
            Assert.assertEquals((double)567.45, (double)event.doubleValue(), (double)0.01);
            this.assertThrowCCE((JsonEvent)event, JsonEvent::stringValue, JsonEvent::booleanValue, JsonEvent::binaryValue, JsonEvent::instantValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
    }

    @Test
    public void testBigInteger() {
        String expected = "18446744073709551615";
        this.testValue(expected, (Handler<JsonEvent>)((Handler)event -> {
            BigInteger big = new BigInteger(expected);
            Assert.assertEquals((Object)big, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            Assert.assertTrue((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            TestCase.assertFalse((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            Assert.assertEquals((long)big.intValue(), (long)event.integerValue().intValue());
            Assert.assertEquals((long)big.longValue(), (long)event.longValue());
            Assert.assertEquals((float)big.floatValue(), (float)event.floatValue().floatValue(), (float)0.01f);
            Assert.assertEquals((double)big.doubleValue(), (double)event.doubleValue(), (double)0.01);
            this.assertThrowCCE((JsonEvent)event, JsonEvent::stringValue, JsonEvent::booleanValue, JsonEvent::binaryValue, JsonEvent::instantValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
    }

    @Test
    public void testBooleanValue() {
        this.testValue("true", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)true, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            Assert.assertTrue((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            Assert.assertTrue((boolean)event.booleanValue());
            this.assertThrowCCE((JsonEvent)event, JsonEvent::integerValue, JsonEvent::longValue, JsonEvent::floatValue, JsonEvent::doubleValue, JsonEvent::stringValue, JsonEvent::binaryValue, JsonEvent::instantValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
        this.testValue("false", (Handler<JsonEvent>)((Handler)event -> {
            Assert.assertEquals((Object)false, (Object)event.value());
            TestCase.assertFalse((boolean)event.isArray());
            TestCase.assertFalse((boolean)event.isObject());
            TestCase.assertFalse((boolean)event.isNumber());
            TestCase.assertFalse((boolean)event.isNull());
            Assert.assertTrue((boolean)event.isBoolean());
            TestCase.assertFalse((boolean)event.isString());
            TestCase.assertFalse((boolean)event.booleanValue());
            this.assertThrowCCE((JsonEvent)event, JsonEvent::integerValue, JsonEvent::longValue, JsonEvent::floatValue, JsonEvent::doubleValue, JsonEvent::stringValue, JsonEvent::binaryValue, JsonEvent::instantValue, JsonEvent::objectValue, JsonEvent::arrayValue);
        }));
    }

    private void testValue(String jsonValue, Handler<JsonEvent> checker) {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.handler(event -> {
            switch (status.getAndIncrement()) {
                case 0: {
                    Assert.assertEquals((Object)JsonEventType.START_OBJECT, (Object)event.type());
                    Assert.assertNull((Object)event.fieldName());
                    Assert.assertNull((Object)event.value());
                    break;
                }
                case 1: {
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    checker.handle(event);
                    Assert.assertEquals((Object)"foo", (Object)event.fieldName());
                    break;
                }
                case 2: {
                    Assert.assertEquals((Object)JsonEventType.END_OBJECT, (Object)event.type());
                    Assert.assertNull((Object)event.fieldName());
                    Assert.assertNull((Object)event.value());
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)("{\"foo\":" + jsonValue + "}")));
        Assert.assertEquals((long)3L, (long)status.get());
    }

    @Test
    public void testParseObjectValueMembers() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.handler(event -> {
            switch (status.getAndIncrement()) {
                case 0: {
                    Assert.assertEquals((Object)JsonEventType.START_OBJECT, (Object)event.type());
                    parser.objectValueMode();
                    break;
                }
                case 1: {
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    Assert.assertTrue((boolean)event.isObject());
                    Assert.assertEquals((Object)new JsonObject(), (Object)event.value());
                    Assert.assertEquals((Object)"foo", (Object)event.fieldName());
                    break;
                }
                case 2: {
                    Assert.assertEquals((Object)"bar", (Object)event.fieldName());
                    Assert.assertTrue((boolean)event.isObject());
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    Assert.assertEquals((Object)new JsonObject(), (Object)event.value());
                    break;
                }
                case 3: {
                    Assert.assertEquals((Object)JsonEventType.END_OBJECT, (Object)event.type());
                    break;
                }
                default: {
                    Assert.fail();
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"{\"foo\":{},\"bar\":{}}"));
        Assert.assertEquals((long)4L, (long)status.get());
    }

    @Test
    public void testParseObjectValueList() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger status = new AtomicInteger();
        parser.objectValueMode();
        parser.handler(event -> {
            switch (status.getAndIncrement()) {
                case 0: {
                    Assert.assertEquals((Object)JsonEventType.START_ARRAY, (Object)event.type());
                    break;
                }
                case 1: {
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    Assert.assertTrue((boolean)event.isObject());
                    Assert.assertEquals((Object)new JsonObject().put("one", (Object)1), (Object)event.value());
                    break;
                }
                case 2: {
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    Assert.assertTrue((boolean)event.isObject());
                    Assert.assertEquals((Object)new JsonObject().put("two", (Object)2), (Object)event.value());
                    break;
                }
                case 3: {
                    Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
                    Assert.assertTrue((boolean)event.isObject());
                    Assert.assertEquals((Object)new JsonObject().put("three", (Object)3), (Object)event.value());
                    break;
                }
                case 4: {
                    Assert.assertEquals((Object)JsonEventType.END_ARRAY, (Object)event.type());
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"[{\"one\":1},{\"two\":2},{\"three\":3}]"));
        Assert.assertEquals((long)5L, (long)status.get());
    }

    @Test
    public void testObjectHandlerScope() {
        JsonParser parser = JsonParser.newParser();
        ArrayList objects = new ArrayList();
        AtomicInteger ends = new AtomicInteger();
        AtomicBoolean obj = new AtomicBoolean();
        parser.handler(event -> {
            switch (event.type()) {
                case START_OBJECT: {
                    parser.objectValueMode();
                    break;
                }
                case VALUE: {
                    if (!obj.get()) break;
                    objects.add((JsonObject)event.value());
                    break;
                }
                case END_OBJECT: {
                    ends.incrementAndGet();
                    obj.set(true);
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"[{\"one\":1},{\"two\":2},{\"three\":3}]"));
        Assert.assertEquals((long)1L, (long)ends.get());
        Assert.assertEquals(Arrays.asList(new JsonObject().put("two", (Object)2), new JsonObject().put("three", (Object)3)), objects);
    }

    @Test
    public void testParseTopValues() {
        HashMap<String, Object> tests = new HashMap<String, Object>();
        tests.put("\"a-string\"", "a-string");
        tests.put("true", true);
        tests.put("false", false);
        tests.put("1234", 1234L);
        tests.put("9223372036854775807", Long.MAX_VALUE);
        tests.forEach((test, expected) -> {
            JsonParser parser = JsonParser.newParser();
            ArrayList values = new ArrayList();
            parser.handler(event -> values.add(event.value()));
            parser.handle((Object)Buffer.buffer((String)test));
            parser.end();
            Assert.assertEquals(Collections.singletonList(expected), values);
        });
    }

    @Test
    public void testObjectMapping() {
        JsonParser parser = JsonParser.newParser();
        ArrayList values = new ArrayList();
        parser.objectValueMode();
        parser.pause();
        parser.handler(event -> values.add(event.mapTo(TheObject.class)));
        parser.handle((Object)Buffer.buffer((String)"{\"f\":\"the-value-1\"}{\"f\":\"the-value-2\"}"));
        Assert.assertEquals(Collections.emptyList(), values);
        parser.fetch(1L);
        Assert.assertEquals(Collections.singletonList(new TheObject("the-value-1")), values);
        parser.fetch(1L);
        Assert.assertEquals(Arrays.asList(new TheObject("the-value-1"), new TheObject("the-value-2")), values);
    }

    @Test
    public void testObjectMappingError() {
        ArrayList values = new ArrayList();
        ArrayList errors = new ArrayList();
        JsonParser.newParser().objectValueMode().handler(event -> values.add(event.mapTo(TheObject.class))).exceptionHandler(errors::add).write(Buffer.buffer((String)"{\"destination\":\"unknown\"}")).end();
        Assert.assertEquals(Collections.emptyList(), values);
        Assert.assertEquals((long)1L, (long)errors.size());
        try {
            JsonParser.newParser().objectValueMode().handler(event -> values.add(event.mapTo(TheObject.class))).write(Buffer.buffer((String)"{\"destination\":\"unknown\"}")).end();
            Assert.fail();
        }
        catch (DecodeException decodeException) {
            // empty catch block
        }
        Assert.assertEquals(Collections.emptyList(), values);
        Assert.assertEquals((long)1L, (long)errors.size());
    }

    @Test
    public void testArrayMapping() {
        JsonParser parser = JsonParser.newParser();
        ArrayList values = new ArrayList();
        parser.arrayValueMode();
        parser.handler(event -> values.add(event.mapTo(LinkedList.class)));
        parser.handle((Object)new JsonArray().add((Object)0).add((Object)1).add((Object)2).toBuffer());
        Assert.assertEquals(Collections.singletonList(Arrays.asList(0L, 1L, 2L)), values);
        Assert.assertEquals(LinkedList.class, values.get(0).getClass());
    }

    @Test
    public void testArrayMappingError() {
        ArrayList values = new ArrayList();
        ArrayList errors = new ArrayList();
        JsonParser.newParser().arrayValueMode().handler(event -> values.add(event.mapTo(TheObject.class))).exceptionHandler(errors::add).write(Buffer.buffer((String)"[]")).end();
        Assert.assertEquals(Collections.emptyList(), values);
        Assert.assertEquals((long)1L, (long)errors.size());
        try {
            JsonParser.newParser().arrayValueMode().handler(event -> values.add(event.mapTo(TheObject.class))).write(Buffer.buffer((String)"[]")).end();
            Assert.fail();
        }
        catch (DecodeException decodeException) {
            // empty catch block
        }
        Assert.assertEquals(Collections.emptyList(), values);
        Assert.assertEquals((long)1L, (long)errors.size());
    }

    @Test
    public void testParseConcatedJSONStream() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger startCount = new AtomicInteger();
        AtomicInteger endCount = new AtomicInteger();
        parser.handler(event -> {
            switch (event.type()) {
                case START_OBJECT: {
                    startCount.incrementAndGet();
                    break;
                }
                case END_OBJECT: {
                    endCount.incrementAndGet();
                    break;
                }
                default: {
                    Assert.fail();
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"{}{}"));
        Assert.assertEquals((long)2L, (long)startCount.get());
        Assert.assertEquals((long)2L, (long)endCount.get());
    }

    @Test
    public void testParseLineDelimitedJSONStream() {
        JsonParser parser = JsonParser.newParser();
        AtomicInteger startCount = new AtomicInteger();
        AtomicInteger endCount = new AtomicInteger();
        AtomicInteger numberCount = new AtomicInteger();
        AtomicInteger nullCount = new AtomicInteger();
        AtomicInteger stringCount = new AtomicInteger();
        parser.handler(event -> {
            switch (event.type()) {
                case START_OBJECT: {
                    startCount.incrementAndGet();
                    break;
                }
                case END_OBJECT: {
                    endCount.incrementAndGet();
                    break;
                }
                case VALUE: {
                    if (event.isNull()) {
                        nullCount.incrementAndGet();
                        break;
                    }
                    if (event.isNumber()) {
                        numberCount.incrementAndGet();
                        break;
                    }
                    if (event.isString()) {
                        stringCount.incrementAndGet();
                        break;
                    }
                    Assert.fail((String)("Unexpected " + String.valueOf(event.type())));
                    break;
                }
                default: {
                    Assert.fail((String)("Unexpected " + String.valueOf(event.type())));
                }
            }
        });
        parser.handle((Object)Buffer.buffer((String)"{}\r\n1\r\nnull\r\n\"foo\""));
        Assert.assertEquals((long)1L, (long)startCount.get());
        Assert.assertEquals((long)1L, (long)endCount.get());
        Assert.assertEquals((long)1L, (long)numberCount.get());
        Assert.assertEquals((long)1L, (long)nullCount.get());
        Assert.assertEquals((long)1L, (long)stringCount.get());
    }

    @Test
    public void testStreamHandle() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        stream.handle("{}");
        TestCase.assertFalse((boolean)stream.isPaused());
        Assert.assertEquals((long)2L, (long)events.size());
    }

    @Test
    public void testStreamPause() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        parser.pause();
        stream.handle("1234");
        Assert.assertTrue((boolean)stream.isPaused());
        Assert.assertEquals((long)0L, (long)events.size());
    }

    @Test
    public void testStreamResume() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        parser.pause();
        stream.handle("{}");
        parser.resume();
        Assert.assertEquals((long)2L, (long)events.size());
        TestCase.assertFalse((boolean)stream.isPaused());
    }

    @Test
    public void testStreamFetch() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        parser.pause();
        stream.handle("{}");
        parser.fetch(1L);
        Assert.assertEquals((long)1L, (long)events.size());
        Assert.assertTrue((boolean)stream.isPaused());
    }

    @Test
    public void testStreamFetchNames() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        parser.pause();
        stream.handle("{\"foo\":\"bar\"}");
        parser.fetch(3L);
        Assert.assertEquals((long)3L, (long)events.size());
        Assert.assertTrue((boolean)stream.isPaused());
    }

    @Test
    public void testStreamPauseInHandler() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(event -> {
            Assert.assertTrue((boolean)events.isEmpty());
            events.add(event);
            parser.pause();
        });
        stream.handle("{}");
        Assert.assertEquals((long)1L, (long)events.size());
        Assert.assertTrue((boolean)stream.isPaused());
    }

    @Test
    public void testStreamFetchInHandler() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(event -> {
            events.add(event);
            stream.fetch(1L);
        });
        stream.pause();
        stream.fetch(1L);
        stream.handle("{}");
        Assert.assertEquals((long)2L, (long)events.size());
        TestCase.assertFalse((boolean)stream.isPaused());
    }

    @Test
    public void testStreamEnd() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        AtomicInteger ended = new AtomicInteger();
        parser.endHandler(v -> ended.incrementAndGet());
        stream.end();
        Assert.assertEquals((long)0L, (long)events.size());
        Assert.assertEquals((long)1L, (long)ended.get());
        Assert.assertEquals((long)0L, (long)stream.pauseCount());
        Assert.assertEquals((long)0L, (long)stream.resumeCount());
    }

    @Test
    public void testStreamPausedEnd() {
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        ArrayList events = new ArrayList();
        parser.handler(events::add);
        AtomicInteger ended = new AtomicInteger();
        parser.endHandler(v -> ended.incrementAndGet());
        parser.pause();
        stream.handle("{}");
        stream.end();
        Assert.assertEquals((long)0L, (long)ended.get());
        Assert.assertEquals((long)0L, (long)events.size());
        parser.fetch(1L);
        Assert.assertEquals((long)1L, (long)events.size());
        Assert.assertEquals((long)0L, (long)ended.get());
        parser.fetch(1L);
        Assert.assertEquals((long)2L, (long)events.size());
        Assert.assertEquals((long)1L, (long)ended.get());
    }

    @Test
    public void testPauseAndResumeInHandler() throws Exception {
        AtomicInteger counter = new AtomicInteger(0);
        FakeStream stream = new FakeStream();
        JsonParser parser = JsonParser.newParser((ReadStream)stream);
        parser.objectValueMode();
        parser.handler(event -> {
            parser.pause();
            Assert.assertNotNull((Object)event);
            Assert.assertEquals((Object)JsonEventType.VALUE, (Object)event.type());
            counter.incrementAndGet();
            parser.resume();
        });
        parser.exceptionHandler(t -> {
            throw new AssertionError(t);
        });
        CountDownLatch latch = new CountDownLatch(1);
        parser.endHandler(end -> {
            Assert.assertEquals((long)4L, (long)counter.get());
            latch.countDown();
        });
        stream.handle("{\"field_0\": \"value");
        stream.handle("_0\"}{\"field_1\": \"value_1\"}");
        stream.handle("{\"field_2\": \"val");
        stream.handle("ue_2\"}{\"field_3\": \"value_3\"}");
        stream.end();
        Assert.assertTrue((boolean)latch.await(5L, TimeUnit.SECONDS));
    }

    @Test
    public void testStreamResume3886() {
        JsonParser parser = JsonParser.newParser();
        ArrayList events = new ArrayList();
        parser.handler(event -> events.add(event));
        parser.pause();
        Buffer b = Buffer.buffer((String)"{ \"a\":\"y\" }");
        parser.handle((Object)b);
        parser.handle((Object)b);
        parser.resume();
        parser.end();
        Assert.assertEquals((long)6L, (long)events.size());
    }

    public static class TheObject {
        private String f;

        public TheObject() {
        }

        public TheObject(String f) {
            this.f = f;
        }

        public void setF(String f) {
            this.f = f;
        }

        public boolean equals(Object obj) {
            TheObject that = (TheObject)obj;
            return Objects.equals(this.f, that.f);
        }
    }
}

