/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.tests.timer;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Context;
import io.vertx.core.Deployable;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Handler;
import io.vertx.core.ThreadingModel;
import io.vertx.core.Timer;
import io.vertx.core.Vertx;
import io.vertx.core.impl.VertxImpl;
import io.vertx.core.internal.ContextInternal;
import io.vertx.core.internal.VertxInternal;
import io.vertx.test.core.Repeat;
import io.vertx.test.core.VertxTestBase;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import org.junit.Test;

public class TimerTest
extends VertxTestBase {
    @Test
    public void testTimer() {
        this.timer(1L);
    }

    @Test
    public void testPeriodic1() {
        this.periodic(new PeriodicArg(100L, 100L), (delay, handler) -> this.vertx.setPeriodic(delay.delay, handler));
    }

    @Test
    public void testPeriodic2() {
        this.periodic(new PeriodicArg(100L, 100L), (delay, handler) -> this.vertx.setPeriodic(delay.delay, delay.delay, handler));
    }

    @Test
    public void testPeriodicWithInitialDelay1() {
        this.periodic(new PeriodicArg(0L, 100L), (delay, handler) -> this.vertx.setPeriodic(delay.initialDelay, delay.delay, handler));
    }

    @Test
    public void testPeriodicWithInitialDelay2() {
        this.periodic(new PeriodicArg(100L, 200L), (delay, handler) -> this.vertx.setPeriodic(delay.initialDelay, delay.delay, handler));
    }

    @Test
    public void testTimings() {
        long start = System.currentTimeMillis();
        long delay = 2000L;
        this.vertx.setTimer(2000L, timerID -> {
            long dur = System.currentTimeMillis() - start;
            this.assertTrue(dur >= 2000L);
            long maxDelay = 4000L;
            this.assertTrue("Timer accuracy: " + dur + " vs " + maxDelay, dur < maxDelay);
            this.vertx.cancelTimer(timerID.longValue());
            this.testComplete();
        });
        this.await();
    }

    @Test
    public void testInVerticle() {
        class MyVerticle
        extends AbstractVerticle {
            AtomicInteger cnt = new AtomicInteger();

            MyVerticle() {
            }

            public void start() {
                Thread thr = Thread.currentThread();
                this.vertx.setTimer(1L, id -> {
                    TimerTest.this.assertSame(thr, Thread.currentThread());
                    if (this.cnt.incrementAndGet() == 5) {
                        TimerTest.this.testComplete();
                    }
                });
                this.vertx.setPeriodic(2L, id -> {
                    TimerTest.this.assertSame(thr, Thread.currentThread());
                    if (this.cnt.incrementAndGet() == 5) {
                        TimerTest.this.testComplete();
                    }
                });
                this.vertx.setPeriodic(3L, 4L, id -> {
                    TimerTest.this.assertSame(thr, Thread.currentThread());
                    if (this.cnt.incrementAndGet() == 5) {
                        TimerTest.this.testComplete();
                    }
                });
            }
        }
        MyVerticle verticle = new MyVerticle();
        this.vertx.deployVerticle((Deployable)verticle);
        this.await();
    }

    private void periodic(final PeriodicArg delay, BiFunction<PeriodicArg, Handler<Long>, Long> abc) {
        int numFires = 10;
        final AtomicLong id = new AtomicLong(-1L);
        final long now = System.currentTimeMillis();
        id.set(abc.apply(delay, new Handler<Long>(){
            int count;

            public void handle(Long timerID) {
                TimerTest.this.assertTrue(System.currentTimeMillis() - now >= delay.initialDelay + (long)this.count * delay.delay);
                TimerTest.this.assertEquals(id.get(), timerID);
                ++this.count;
                if (this.count == 10) {
                    TimerTest.this.vertx.cancelTimer(timerID.longValue());
                    TimerTest.this.setEndTimer();
                }
                if (this.count > 10) {
                    TimerTest.this.fail("Fired too many times");
                }
            }
        }));
        this.await();
    }

    private void timer(long delay) {
        final AtomicLong id = new AtomicLong(-1L);
        id.set(this.vertx.setTimer(delay, (Handler)new Handler<Long>(){
            int count;
            boolean fired;

            public void handle(Long timerID) {
                TimerTest.this.assertFalse(this.fired);
                this.fired = true;
                TimerTest.this.assertEquals(id.get(), timerID);
                TimerTest.this.assertEquals(0L, this.count);
                ++this.count;
                TimerTest.this.setEndTimer();
            }
        }));
        this.await();
    }

    private void setEndTimer() {
        this.vertx.setTimer(10L, id -> this.testComplete());
    }

    @Test
    public void testCancelTimerWhenScheduledOnWorker() {
        this.vertx.deployVerticle((Deployable)new AbstractVerticle(){

            public void start() throws Exception {
                long id = this.vertx.setTimer(100L, id_ -> TimerTest.this.fail());
                Thread.sleep(200L);
                TimerTest.this.assertTrue(this.vertx.cancelTimer(id));
                TimerTest.this.testComplete();
            }
        }, new DeploymentOptions().setThreadingModel(ThreadingModel.WORKER));
        this.await();
    }

    @Test
    public void testWorkerTimer() {
        this.vertx.deployVerticle((Deployable)new AbstractVerticle(){

            public void start() throws Exception {
                this.vertx.setTimer(10L, id -> {
                    TimerTest.this.assertTrue(Context.isOnWorkerThread());
                    TimerTest.this.testComplete();
                });
            }
        }, new DeploymentOptions().setThreadingModel(ThreadingModel.WORKER));
        this.await();
    }

    @Test
    public void testFailInTimer() {
        RuntimeException failure = new RuntimeException();
        Context ctx = this.vertx.getOrCreateContext();
        ctx.runOnContext(v -> {
            ctx.exceptionHandler(err -> {
                this.assertSame(err, failure);
                this.testComplete();
            });
            this.vertx.setTimer(5L, id -> {
                throw failure;
            });
        });
        this.await();
    }

    @Test
    public void testCancellationRace() throws Exception {
        for (int i = 0; i < 200; ++i) {
            AtomicBoolean fired = new AtomicBoolean();
            long timerId = this.vertx.setTimer(5L, id -> fired.set(true));
            Thread.sleep(5L);
            boolean res = this.vertx.cancelTimer(timerId);
            if (res && fired.get()) {
                throw new AssertionError((Object)("It failed " + i));
            }
        }
    }

    @Test
    public void testUndeployCancelTimer() {
        this.testUndeployCancellation(() -> this.vertx.setTimer(1000L, id -> {}));
    }

    @Test
    public void testUndeployCancelPeriodic() {
        this.testUndeployCancellation(() -> this.vertx.setPeriodic(1000L, id -> {}));
    }

    private void testUndeployCancellation(final Supplier<Long> f) {
        final AtomicLong timer = new AtomicLong();
        this.vertx.deployVerticle((Deployable)new AbstractVerticle(){

            public void start() {
                timer.set((Long)f.get());
            }
        }).compose(deployment -> this.vertx.undeploy(deployment)).onComplete(this.onSuccess(v -> {
            this.assertFalse(this.vertx.cancelTimer(timer.get()));
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testTimerOnContext() {
        this.disableThreadChecks();
        ContextInternal ctx1 = ((VertxInternal)this.vertx).createEventLoopContext();
        this.waitFor(2);
        ContextInternal ctx2 = ((VertxInternal)this.vertx).createEventLoopContext();
        this.assertNotSame(ctx1, ctx2);
        ctx2.runOnContext(v -> {
            this.vertx.setTimer(10L, l -> {
                this.assertSame(ctx2, this.vertx.getOrCreateContext());
                this.complete();
            });
            ctx1.setTimer(10L, l -> {
                this.assertSame(ctx1, this.vertx.getOrCreateContext());
                this.complete();
            });
        });
        this.await();
    }

    @Test
    public void testPeriodicOnContext() {
        this.testPeriodicOnContext(((VertxInternal)this.vertx).createEventLoopContext());
    }

    @Test
    public void testPeriodicOnDuplicatedContext() {
        this.testPeriodicOnContext(((VertxInternal)this.vertx).createEventLoopContext().duplicate());
    }

    private void testPeriodicOnContext(final ContextInternal ctx2) {
        this.disableThreadChecks();
        this.waitFor(4);
        final ContextInternal ctx1 = ((VertxInternal)this.vertx).createEventLoopContext();
        this.assertNotSame(ctx1, ctx2);
        ctx2.runOnContext(v -> {
            final Thread th = Thread.currentThread();
            this.vertx.setPeriodic(10L, (Handler)new Handler<Long>(){
                int count;

                public void handle(Long l) {
                    TimerTest.this.assertSame(th, Thread.currentThread());
                    ContextInternal current = (ContextInternal)TimerTest.this.vertx.getOrCreateContext();
                    TimerTest.this.assertNotNull(current);
                    TimerTest.this.assertTrue(current.isDuplicate());
                    TimerTest.this.assertNotSame(ctx2, current);
                    TimerTest.this.assertSame(ctx2.unwrap(), current.unwrap());
                    if (++this.count == 2) {
                        TimerTest.this.vertx.cancelTimer(l.longValue());
                    }
                    TimerTest.this.complete();
                }
            });
            ctx1.setPeriodic(10L, (Handler)new Handler<Long>(){
                int count;

                public void handle(Long l) {
                    ContextInternal current = (ContextInternal)TimerTest.this.vertx.getOrCreateContext();
                    TimerTest.this.assertNotNull(current);
                    TimerTest.this.assertTrue(current.isDuplicate());
                    TimerTest.this.assertNotSame(ctx1, current);
                    TimerTest.this.assertSame(ctx1, current.unwrap());
                    if (++this.count == 2) {
                        TimerTest.this.vertx.cancelTimer(l.longValue());
                    }
                    TimerTest.this.complete();
                }
            });
        });
        this.await();
    }

    @Repeat(times=100)
    @Test
    public void testRaceWhenTimerCreatedOutsideEventLoop() {
        int numThreads = 1000;
        int numIter = 1;
        Thread[] threads = new Thread[numThreads];
        AtomicInteger count = new AtomicInteger(numIter * numThreads);
        for (int i = 0; i < numThreads; ++i) {
            Thread th = new Thread(() -> ((VertxImpl)this.vertx).scheduleTimeout(((VertxImpl)this.vertx).getOrCreateContext(), false, 1L, TimeUnit.NANOSECONDS, false, ignore -> count.decrementAndGet()));
            th.start();
            threads[i] = th;
        }
        TimerTest.waitUntil(() -> count.get() == 0);
    }

    @Test
    public void testContextTimer() {
        this.waitFor(2);
        this.vertx.deployVerticle((Deployable)new AbstractVerticle(){

            public void start() throws Exception {
                ((ContextInternal)this.context).setTimer(1000L, id -> TimerTest.this.complete());
                this.context.runOnContext(v -> this.vertx.undeploy(this.context.deploymentID()).onComplete(TimerTest.this.onSuccess(ar -> ((ContextInternal)this.context).setTimer(1L, id -> TimerTest.this.complete()))));
            }
        });
        this.await();
    }

    @Test
    public void testTimerFire() {
        long now = System.currentTimeMillis();
        Timer timer = this.vertx.timer(1L, TimeUnit.SECONDS);
        timer.onComplete(this.onSuccess(v -> {
            this.assertTrue(System.currentTimeMillis() - now >= 800L);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testTimerFireOnContext1() {
        new Thread(() -> {
            ContextInternal ctx = (ContextInternal)this.vertx.getOrCreateContext();
            Timer timer = this.vertx.timer(10L, TimeUnit.MILLISECONDS);
            timer.onComplete(this.onSuccess(v -> {
                this.assertSame(ctx.nettyEventLoop(), ((ContextInternal)Vertx.currentContext()).nettyEventLoop());
                this.testComplete();
            }));
        }).start();
        this.await();
    }

    @Test
    public void testTimerFireOnContext2() {
        this.vertx.runOnContext(v1 -> {
            Context current = this.vertx.getOrCreateContext();
            ContextInternal context = ((VertxInternal)this.vertx).createEventLoopContext();
            this.assertNotSame(context, current);
            Timer timer = context.timer(10L, TimeUnit.MILLISECONDS);
            timer.onComplete(this.onSuccess(v2 -> {
                this.assertSame(context, Vertx.currentContext());
                this.testComplete();
            }));
        });
        this.await();
    }

    @Test
    public void testFailTimerTaskWhenCancellingTimer() {
        Timer timer = this.vertx.timer(10000L);
        this.assertTrue(timer.cancel());
        TimerTest.waitUntil(() -> ((Timer)timer).failed());
        this.assertTrue(timer.cause() instanceof CancellationException);
    }

    @Test
    public void testFailTimerTaskWhenClosingVertx() throws Exception {
        Vertx vertx = Vertx.vertx();
        Timer timer = vertx.timer(10000L);
        this.awaitFuture(vertx.close());
        TimerTest.waitUntil(() -> ((Timer)timer).failed());
        this.assertTrue(timer.cause() instanceof CancellationException);
    }

    static class PeriodicArg {
        final long initialDelay;
        final long delay;

        PeriodicArg(long initialDelay, long delay) {
            this.initialDelay = initialDelay;
            this.delay = delay;
        }
    }
}

