/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.groovy.ext.dropwizard;
import groovy.transform.CompileStatic
import io.vertx.lang.groovy.InternalHelper
import io.vertx.core.json.JsonObject
import io.vertx.groovy.core.metrics.Measured
import io.vertx.groovy.core.Vertx
import java.util.Set
import io.vertx.core.json.JsonObject
/**
 * The metrics service mainly allows to return a snapshot of measured objects.
*/
@CompileStatic
public class MetricsService {
  private final def io.vertx.ext.dropwizard.MetricsService delegate;
  public MetricsService(Object delegate) {
    this.delegate = (io.vertx.ext.dropwizard.MetricsService) delegate;
  }
  public Object getDelegate() {
    return delegate;
  }
  /**
   * Creates a metric service for a given {@link io.vertx.groovy.core.Vertx} instance.
   * @param vertx the vertx instance
   * @return the metrics service
   */
  public static MetricsService create(Vertx vertx) {
    def ret= InternalHelper.safeCreate(io.vertx.ext.dropwizard.MetricsService.create((io.vertx.core.Vertx)vertx.getDelegate()), io.vertx.groovy.ext.dropwizard.MetricsService.class);
    return ret;
  }
  /**
   * @param measured the measure object
   * @param measured 
   * @return the base name of the measured object
   */
  public String getBaseName(Measured measured) {
    def ret = this.delegate.getBaseName((io.vertx.core.metrics.Measured)measured.getDelegate());
    return ret;
  }
  /**
   * @return the known metrics names by this service
   * @return 
   */
  public Set<String> metricsNames() {
    def ret = this.delegate.metricsNames();
    return ret;
  }
  /**
   * Will return the metrics that correspond with the <code>measured</code> object, null if no metrics is available.<p/>
   *
   * Note: in the case of scaled servers, the JsonObject returns an aggregation of the metrics as the
   * dropwizard backend reports to a single server.
   * @param measured 
   * @return the map of metrics where the key is the name of the metric (excluding the base name unless for the Vert.x object) and the value is the json data representing that metric
   */
  public Map<String, Object> getMetricsSnapshot(Measured measured) {
    def ret = (Map<String, Object>)InternalHelper.wrapObject(this.delegate.getMetricsSnapshot((io.vertx.core.metrics.Measured)measured.getDelegate()));
    return ret;
  }
  /**
   * Will return the metrics that begins with the <code>baseName</code>, null if no metrics is available.<p/>
   *
   * Note: in the case of scaled servers, the JsonObject returns an aggregation of the metrics as the
   * dropwizard backend reports to a single server.
   * @param baseName 
   * @return the map of metrics where the key is the name of the metric and the value is the json data representing that metric
   */
  public Map<String, Object> getMetricsSnapshot(String baseName) {
    def ret = (Map<String, Object>)InternalHelper.wrapObject(this.delegate.getMetricsSnapshot(baseName));
    return ret;
  }
}
