/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.jwt;

import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.jwt.Crypto;
import io.vertx.ext.jwt.impl.SignatureHelper;
import java.io.ByteArrayInputStream;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.AlgorithmParameters;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EncodedKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.UUID;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

public final class JWK
implements Crypto {
    private static final Charset UTF8 = StandardCharsets.UTF_8;
    private final String kid;
    private String alg;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private Signature signature;
    private Cipher cipher;
    private X509Certificate certificate;
    private Mac mac;
    private boolean symmetric;
    private boolean ecdsa;
    private int ecdsaLength;

    public JWK(String algorithm, String pemPub, String pemSec) {
        this(algorithm, false, pemPub, pemSec);
    }

    public JWK(String algorithm, boolean isCertificate, String pemPub, String pemSec) {
        try {
            EncodedKeySpec keyspec;
            KeyFactory kf;
            HashMap<String, String> alias = new HashMap<String, String>(){
                {
                    this.put("RS256", "SHA256withRSA");
                    this.put("RS384", "SHA384withRSA");
                    this.put("RS512", "SHA512withRSA");
                    this.put("ES256", "SHA256withECDSA");
                    this.put("ES384", "SHA384withECDSA");
                    this.put("ES512", "SHA512withECDSA");
                }
            };
            switch (algorithm) {
                case "RS256": 
                case "RS384": 
                case "RS512": {
                    kf = KeyFactory.getInstance("RSA");
                    break;
                }
                case "ES256": 
                case "ES384": 
                case "ES512": {
                    kf = KeyFactory.getInstance("EC");
                    this.ecdsa = true;
                    this.ecdsaLength = this.ECDSALength((String)alias.get(algorithm));
                    break;
                }
                default: {
                    throw new RuntimeException("Unknown algorithm factory for: " + algorithm);
                }
            }
            this.alg = algorithm;
            this.kid = algorithm + (pemPub != null ? Integer.valueOf(pemPub.hashCode()) : "") + "-" + (pemSec != null ? Integer.valueOf(pemSec.hashCode()) : "");
            if (pemPub != null) {
                if (isCertificate) {
                    CertificateFactory cf = CertificateFactory.getInstance("X.509");
                    this.certificate = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(pemPub.getBytes(UTF8)));
                } else {
                    keyspec = new X509EncodedKeySpec(Base64.getMimeDecoder().decode(pemPub));
                    this.publicKey = kf.generatePublic(keyspec);
                }
            }
            if (pemSec != null) {
                keyspec = new PKCS8EncodedKeySpec(Base64.getMimeDecoder().decode(pemSec));
                this.privateKey = kf.generatePrivate(keyspec);
            }
            this.signature = Signature.getInstance((String)alias.get(this.alg));
        }
        catch (NoSuchAlgorithmException | CertificateException | InvalidKeySpecException e) {
            throw new RuntimeException(e);
        }
    }

    public JWK(String algorithm, String hmac) {
        try {
            HashMap<String, String> alias = new HashMap<String, String>(){
                {
                    this.put("HS256", "HMacSHA256");
                    this.put("HS384", "HMacSHA384");
                    this.put("HS512", "HMacSHA512");
                }
            };
            this.alg = algorithm;
            if (!alias.containsKey(this.alg)) {
                throw new NoSuchAlgorithmException(this.alg);
            }
            this.kid = algorithm + hmac.hashCode();
            this.mac = Mac.getInstance((String)alias.get(this.alg));
            this.mac.init(new SecretKeySpec(hmac.getBytes(UTF8), (String)alias.get(this.alg)));
            this.symmetric = true;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    public JWK(JsonObject json) {
        this.kid = json.getString("kid", UUID.randomUUID().toString());
        try {
            switch (json.getString("kty")) {
                case "RSA": {
                    this.createRSA(json);
                    break;
                }
                case "EC": {
                    this.createEC(json);
                    break;
                }
                case "oct": {
                    this.createOCT(json);
                    break;
                }
                default: {
                    throw new RuntimeException("Unsupported key type: " + json.getString("kty"));
                }
            }
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | CertificateException | InvalidKeySpecException | InvalidParameterSpecException | NoSuchPaddingException e) {
            throw new RuntimeException(e);
        }
    }

    private void createRSA(JsonObject json) throws NoSuchAlgorithmException, InvalidKeySpecException, CertificateException, NoSuchPaddingException {
        BigInteger e;
        BigInteger n;
        HashMap<String, String> alias = new HashMap<String, String>(){
            {
                this.put("RS256", "SHA256withRSA");
                this.put("RS384", "SHA384withRSA");
                this.put("RS512", "SHA512withRSA");
            }
        };
        this.alg = json.getString("alg", "RS256");
        if (!alias.containsKey(this.alg)) {
            throw new NoSuchAlgorithmException(this.alg);
        }
        if (JWK.jsonHasProperties(json, "n", "e")) {
            n = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("n")));
            e = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("e")));
            this.publicKey = KeyFactory.getInstance("RSA").generatePublic(new RSAPublicKeySpec(n, e));
        }
        if (JWK.jsonHasProperties(json, "n", "e", "d", "p", "q", "dp", "dq", "qi")) {
            n = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("n")));
            e = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("e")));
            BigInteger d = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("d")));
            BigInteger p = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("p")));
            BigInteger q = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("q")));
            BigInteger dp = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("dp")));
            BigInteger dq = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("dq")));
            BigInteger qi = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("qi")));
            this.privateKey = KeyFactory.getInstance("RSA").generatePrivate(new RSAPrivateCrtKeySpec(n, e, d, p, q, dp, dq, qi));
        }
        if (json.containsKey("x5c")) {
            JsonArray x5c = json.getJsonArray("x5c");
            if (x5c.size() > 1) {
                throw new RuntimeException("Certificate Chain length > 1 is not supported");
            }
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            this.certificate = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(this.addBoundaries(x5c.getString(0)).getBytes(UTF8)));
        }
        switch (json.getString("use", "sig")) {
            case "sig": {
                try {
                    this.signature = Signature.getInstance((String)alias.get(this.alg));
                    break;
                }
                catch (NoSuchAlgorithmException e2) {
                    throw new RuntimeException(e2);
                }
            }
            case "enc": {
                this.cipher = Cipher.getInstance("RSA");
            }
        }
    }

    private String addBoundaries(String certificate) {
        return "-----BEGIN CERTIFICATE-----\n" + certificate + "\n-----END CERTIFICATE-----";
    }

    private void createEC(JsonObject json) throws NoSuchAlgorithmException, InvalidKeySpecException, InvalidParameterSpecException, NoSuchPaddingException {
        BigInteger y;
        BigInteger x;
        HashMap<String, String> alias = new HashMap<String, String>(){
            {
                this.put("ES256", "SHA256withECDSA");
                this.put("ES384", "SHA384withECDSA");
                this.put("ES512", "SHA512withECDSA");
            }
        };
        this.alg = json.getString("alg", "ES256");
        this.ecdsa = true;
        if (!alias.containsKey(this.alg)) {
            throw new NoSuchAlgorithmException(this.alg);
        }
        this.ecdsaLength = this.ECDSALength((String)alias.get(this.alg));
        AlgorithmParameters parameters = AlgorithmParameters.getInstance("EC");
        parameters.init(new ECGenParameterSpec(JWK.translate(json.getString("crv"))));
        if (JWK.jsonHasProperties(json, "x", "y")) {
            x = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("x")));
            y = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("y")));
            this.publicKey = KeyFactory.getInstance("EC").generatePublic(new ECPublicKeySpec(new ECPoint(x, y), parameters.getParameterSpec(ECParameterSpec.class)));
        }
        if (JWK.jsonHasProperties(json, "x", "y", "d")) {
            x = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("x")));
            y = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("y")));
            BigInteger d = new BigInteger(1, Base64.getUrlDecoder().decode(json.getString("d")));
            this.privateKey = KeyFactory.getInstance("EC").generatePrivate(new ECPrivateKeySpec(d, parameters.getParameterSpec(ECParameterSpec.class)));
        }
        switch (json.getString("use", "sig")) {
            case "sig": {
                try {
                    this.signature = Signature.getInstance((String)alias.get(this.alg));
                    break;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new RuntimeException(e);
                }
            }
            default: {
                throw new RuntimeException("EC Encryption not supported");
            }
        }
    }

    private void createOCT(JsonObject json) throws NoSuchAlgorithmException, InvalidKeyException, InvalidKeySpecException {
        HashMap<String, String> alias = new HashMap<String, String>(){
            {
                this.put("HS256", "HMacSHA256");
                this.put("HS384", "HMacSHA384");
                this.put("HS512", "HMacSHA512");
            }
        };
        this.alg = json.getString("alg", "HS256");
        if (!alias.containsKey(this.alg)) {
            throw new NoSuchAlgorithmException(this.alg);
        }
        this.mac = Mac.getInstance((String)alias.get(this.alg));
        this.mac.init(new SecretKeySpec(json.getString("k").getBytes(UTF8), (String)alias.get(this.alg)));
        this.symmetric = true;
    }

    public String getAlgorithm() {
        return this.alg;
    }

    @Override
    public String getId() {
        return this.kid;
    }

    public synchronized byte[] encrypt(byte[] payload) {
        if (this.cipher == null) {
            throw new RuntimeException("Key use is not 'enc'");
        }
        try {
            this.cipher.init(1, this.publicKey);
            this.cipher.update(payload);
            return this.cipher.doFinal();
        }
        catch (InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            throw new RuntimeException(e);
        }
    }

    public synchronized byte[] decrypt(byte[] payload) {
        if (this.cipher == null) {
            throw new RuntimeException("Key use is not 'enc'");
        }
        try {
            this.cipher.init(2, this.privateKey);
            this.cipher.update(payload);
            return this.cipher.doFinal();
        }
        catch (InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public synchronized byte[] sign(byte[] payload) {
        if (this.symmetric) {
            return this.mac.doFinal(payload);
        }
        if (this.signature == null) {
            throw new RuntimeException("Key use is not 'sig'");
        }
        try {
            this.signature.initSign(this.privateKey);
            this.signature.update(payload);
            if (this.ecdsa) {
                return SignatureHelper.toJWS(this.signature.sign(), this.ecdsaLength);
            }
            return this.signature.sign();
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public synchronized boolean verify(byte[] expected, byte[] payload) {
        if (this.symmetric) {
            return Arrays.equals(expected, this.sign(payload));
        }
        if (this.signature == null) {
            throw new RuntimeException("Key use is not 'sig'");
        }
        try {
            if (this.publicKey != null) {
                this.signature.initVerify(this.publicKey);
            }
            if (this.certificate != null) {
                this.signature.initVerify(this.certificate);
            }
            this.signature.update(payload);
            if (this.ecdsa) {
                return this.signature.verify(SignatureHelper.toDER(expected));
            }
            return this.signature.verify(expected);
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new RuntimeException(e);
        }
    }

    private static String translate(String crv) {
        switch (crv) {
            case "P-256": {
                return "secp256r1";
            }
            case "P-384": {
                return "secp384r1";
            }
            case "P-521": {
                return "secp521r1";
            }
        }
        return "";
    }

    private static boolean jsonHasProperties(JsonObject json, String ... properties) {
        for (String property : properties) {
            if (json.containsKey(property)) continue;
            return false;
        }
        return true;
    }
}

