/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/buffer */
var utils = require('vertx-js/util/utils');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JBuffer = io.vertx.core.buffer.Buffer;

/**
 A Buffer represents a sequence of zero or more bytes that can be written to or read from, and which expands as
 necessary to accommodate any bytes written to it.<p>
 @class
*/
var Buffer = function(j_val) {

  var j_buffer = j_val;
  var that = this;

  /**
  
   @public
   @param enc {string} 
   @return {string}
   */
  this.toString = function(enc) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_buffer.toString(enc);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getByte = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getByte(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getInt = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getInt(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getLong = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getLong(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getDouble = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getDouble(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getFloat = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getFloat(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @return {number}
   */
  this.getShort = function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_buffer.getShort(pos);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param start {number} 
   @param end {number} 
   @return {Buffer}
   */
  this.getBuffer = function(start, end) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      return new Buffer(j_buffer.getBuffer(start, end));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param start {number} 
   @param end {number} 
   @return {string}
   */
  this.getString = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      return j_buffer.getString(__args[0], __args[1]);
    }  else if (__args.length === 3 && typeof __args[0] ==='number' && typeof __args[1] ==='number' && typeof __args[2] === 'string') {
      return j_buffer.getString(__args[0], __args[1], __args[2]);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param buff {Buffer} 
   @return {Buffer}
   */
  this.appendBuffer = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_buffer.appendBuffer(__args[0]._jdel);
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] ==='number' && typeof __args[2] ==='number') {
      j_buffer.appendBuffer(__args[0]._jdel, __args[1], __args[2]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param b {number} 
   @return {Buffer}
   */
  this.appendByte = function(b) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendByte(b);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param i {number} 
   @return {Buffer}
   */
  this.appendInt = function(i) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendInt(i);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param l {number} 
   @return {Buffer}
   */
  this.appendLong = function(l) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendLong(l);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param s {number} 
   @return {Buffer}
   */
  this.appendShort = function(s) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendShort(s);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param f {number} 
   @return {Buffer}
   */
  this.appendFloat = function(f) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendFloat(f);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param d {number} 
   @return {Buffer}
   */
  this.appendDouble = function(d) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_buffer.appendDouble(d);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param str {string} 
   @return {Buffer}
   */
  this.appendString = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_buffer.appendString(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_buffer.appendString(__args[0], __args[1]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param b {number} 
   @return {Buffer}
   */
  this.setByte = function(pos, b) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setByte(pos, b);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param i {number} 
   @return {Buffer}
   */
  this.setInt = function(pos, i) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setInt(pos, i);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param l {number} 
   @return {Buffer}
   */
  this.setLong = function(pos, l) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setLong(pos, l);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param d {number} 
   @return {Buffer}
   */
  this.setDouble = function(pos, d) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setDouble(pos, d);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param f {number} 
   @return {Buffer}
   */
  this.setFloat = function(pos, f) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setFloat(pos, f);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param s {number} 
   @return {Buffer}
   */
  this.setShort = function(pos, s) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      j_buffer.setShort(pos, s);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param b {Buffer} 
   @return {Buffer}
   */
  this.setBuffer = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'object' && __args[1]._jdel) {
      j_buffer.setBuffer(__args[0], __args[1]._jdel);
      return that;
    }  else if (__args.length === 4 && typeof __args[0] ==='number' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
      j_buffer.setBuffer(__args[0], __args[1]._jdel, __args[2], __args[3]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param pos {number} 
   @param str {string} 
   @return {Buffer}
   */
  this.setString = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'string') {
      j_buffer.setString(__args[0], __args[1]);
      return that;
    }  else if (__args.length === 3 && typeof __args[0] ==='number' && typeof __args[1] === 'string' && typeof __args[2] === 'string') {
      j_buffer.setString(__args[0], __args[1], __args[2]);
      return that;
    } else utils.invalidArgs();
  };

  /**
   Returns the length of the buffer, measured in bytes.
   All positions are indexed from zero.

   @public

   @return {number}
   */
  this.length = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_buffer.length();
    } else utils.invalidArgs();
  };

  /**
   Returns a copy of the entire Buffer.

   @public

   @return {Buffer}
   */
  this.copy = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new Buffer(j_buffer.copy());
    } else utils.invalidArgs();
  };

  /**
   Returns a slice of this buffer. Modifying the content
   of the returned buffer or this buffer affects each other's content
   while they maintain separate indexes and marks.

   @public

   @return {Buffer}
   */
  this.slice = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new Buffer(j_buffer.slice());
    }  else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      return new Buffer(j_buffer.slice(__args[0], __args[1]));
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_buffer;
};

/**

 @memberof module:vertx-js/buffer

 @return {Buffer}
 */
Buffer.buffer = function() {
  var __args = arguments;
  if (__args.length === 0) {
    return new Buffer(JBuffer.buffer());
  }else if (__args.length === 1 && typeof __args[0] ==='number') {
    return new Buffer(JBuffer.buffer(__args[0]));
  }else if (__args.length === 1 && typeof __args[0] === 'string') {
    return new Buffer(JBuffer.buffer(__args[0]));
  }else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
    return new Buffer(JBuffer.buffer(__args[0], __args[1]));
  } else utils.invalidArgs();
};

// We export the Constructor function
module.exports = Buffer;