/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/datagram_socket */
var utils = require('vertx-js/util/utils');
var PacketWritestream = require('vertx-js/packet_writestream');
var Buffer = require('vertx-js/buffer');
var Measured = require('vertx-js/measured');
var DatagramPacket = require('vertx-js/datagram_packet');
var ReadStream = require('vertx-js/read_stream');
var SocketAddress = require('vertx-js/socket_address');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JDatagramSocket = io.vertx.core.datagram.DatagramSocket;

/**

 @class
*/
var DatagramSocket = function(j_val) {

  var j_datagramSocket = j_val;
  var that = this;
  ReadStream.call(this, j_val);
  Measured.call(this, j_val);

  /**
   The metric base name

   @public

   @return {string}
   */
  this.metricBaseName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_datagramSocket.metricBaseName();
    } else utils.invalidArgs();
  };

  /**
   Will return the metrics that correspond with this measured object.

   @public

   @return {Array.<string>}
   */
  this.metrics = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMap(j_datagramSocket.metrics());
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param packet {Buffer} 
   @param port {number} 
   @param host {string} 
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.send = function() {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] ==='number' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_datagramSocket.send(__args[0]._jdel, __args[1], __args[2], function(ar) {
      if (ar.succeeded()) {
        __args[3](new DatagramSocket(ar.result()), null);
      } else {
        __args[3](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] ==='number' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_datagramSocket.send(__args[0], __args[1], __args[2], function(ar) {
      if (ar.succeeded()) {
        __args[3](new DatagramSocket(ar.result()), null);
      } else {
        __args[3](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='number' && typeof __args[3] === 'string' && typeof __args[4] === 'function') {
      j_datagramSocket.send(__args[0], __args[1], __args[2], __args[3], function(ar) {
      if (ar.succeeded()) {
        __args[4](new DatagramSocket(ar.result()), null);
      } else {
        __args[4](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param port {number} 
   @param host {string} 
   @return {PacketWritestream}
   */
  this.sender = function(port, host) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'string') {
      return new PacketWritestream(j_datagramSocket.sender(port, host));
    } else utils.invalidArgs();
  };

  /**
  
   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_datagramSocket.close();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_datagramSocket.close(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {SocketAddress}
   */
  this.localAddress = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedlocalAddress == null) {
        that.cachedlocalAddress = new SocketAddress(j_datagramSocket.localAddress());
      }
      return that.cachedlocalAddress;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param multicastAddress {string} 
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.listenMulticastGroup = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_datagramSocket.listenMulticastGroup(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](new DatagramSocket(ar.result()), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_datagramSocket.listenMulticastGroup(__args[0], __args[1], __args[2], function(ar) {
      if (ar.succeeded()) {
        __args[3](new DatagramSocket(ar.result()), null);
      } else {
        __args[3](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Leaves a multicast group and so stop listen for packets send to it.
   @public
   @param multicastAddress {string} 
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.unlistenMulticastGroup = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_datagramSocket.unlistenMulticastGroup(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](new DatagramSocket(ar.result()), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_datagramSocket.unlistenMulticastGroup(__args[0], __args[1], __args[2], function(ar) {
      if (ar.succeeded()) {
        __args[3](new DatagramSocket(ar.result()), null);
      } else {
        __args[3](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param multicastAddress {string} 
   @param sourceToBlock {string} 
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.blockMulticastGroup = function() {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_datagramSocket.blockMulticastGroup(__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](new DatagramSocket(ar.result()), null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_datagramSocket.blockMulticastGroup(__args[0], __args[1], __args[2], function(ar) {
      if (ar.succeeded()) {
        __args[3](new DatagramSocket(ar.result()), null);
      } else {
        __args[3](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param port {number} 
   @param host {string} 
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.listen = function(port, host, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] ==='number' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_datagramSocket.listen(port, host, function(ar) {
      if (ar.succeeded()) {
        handler(new DatagramSocket(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {DatagramSocket}
   */
  this.pause = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_datagramSocket.pause();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {DatagramSocket}
   */
  this.resume = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_datagramSocket.resume();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param endHandler {function} 
   @return {DatagramSocket}
   */
  this.endHandler = function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_datagramSocket.endHandler(endHandler);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.handler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_datagramSocket.handler(function(jVal) {
      handler(new DatagramPacket(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {DatagramSocket}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_datagramSocket.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_datagramSocket;
};

// We export the Constructor function
module.exports = DatagramSocket;