/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/file_system */
var utils = require('vertx-js/util/utils');
var AsyncFile = require('vertx-js/async_file');
var Buffer = require('vertx-js/buffer');
var FileSystemProps = require('vertx-js/file_system_props');
var FileProps = require('vertx-js/file_props');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JFileSystem = io.vertx.core.file.FileSystem;
var OpenOptions = io.vertx.core.file.OpenOptions;
var OpenOptions = io.vertx.core.file.OpenOptions;

/**
 Contains a broad set of operations for manipulating files.<p>
 A blocking and non blocking version of each operation is provided.<p>
 The non blocking versions take a handler which is called when the operation completes or an error occurs.<p>
 The blocking versions return the results, or throw exceptions directly.<p>
 It is highly recommended the non blocking versions are used unless you are sure the operation
 will not block for a significant period of time.<p>
 Instances of FileSystem are thread-safe.<p>
 @class
*/
var FileSystem = function(j_val) {

  var j_fileSystem = j_val;
  var that = this;

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.copy = function(from, to, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.copy(from, to, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @return {FileSystem}
   */
  this.copyBlocking = function(from, to) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.copyBlocking(from, to);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @param recursive {boolean} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.copyRecursive = function(from, to, recursive, handler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean' && typeof __args[3] === 'function') {
      j_fileSystem.copyRecursive(from, to, recursive, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @param recursive {boolean} 
   @return {FileSystem}
   */
  this.copyRecursiveBlocking = function(from, to, recursive) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_fileSystem.copyRecursiveBlocking(from, to, recursive);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.move = function(from, to, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.move(from, to, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param from {string} 
   @param to {string} 
   @return {FileSystem}
   */
  this.moveBlocking = function(from, to) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.moveBlocking(from, to);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param len {number} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.truncate = function(path, len, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] ==='number' && typeof __args[2] === 'function') {
      j_fileSystem.truncate(path, len, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param len {number} 
   @return {FileSystem}
   */
  this.truncateBlocking = function(path, len) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] ==='number') {
      j_fileSystem.truncateBlocking(path, len);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param perms {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.chmod = function(path, perms, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.chmod(path, perms, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param perms {string} 
   @return {FileSystem}
   */
  this.chmodBlocking = function(path, perms) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.chmodBlocking(path, perms);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param perms {string} 
   @param dirPerms {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.chmodRecursive = function(path, perms, dirPerms, handler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_fileSystem.chmodRecursive(path, perms, dirPerms, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param perms {string} 
   @param dirPerms {string} 
   @return {FileSystem}
   */
  this.chmodRecursiveBlocking = function(path, perms, dirPerms) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string') {
      j_fileSystem.chmodRecursiveBlocking(path, perms, dirPerms);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param user {string} 
   @param group {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.chown = function(path, user, group, handler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_fileSystem.chown(path, user, group, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param user {string} 
   @param group {string} 
   @return {FileSystem}
   */
  this.chownBlocking = function(path, user, group) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string') {
      j_fileSystem.chownBlocking(path, user, group);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.props = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.props(path, function(ar) {
      if (ar.succeeded()) {
        handler(new FileProps(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileProps}
   */
  this.propsBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return new FileProps(j_fileSystem.propsBlocking(path));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.lprops = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.lprops(path, function(ar) {
      if (ar.succeeded()) {
        handler(new FileProps(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileProps}
   */
  this.lpropsBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return new FileProps(j_fileSystem.lpropsBlocking(path));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param existing {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.link = function(link, existing, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.link(link, existing, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param existing {string} 
   @return {FileSystem}
   */
  this.linkBlocking = function(link, existing) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.linkBlocking(link, existing);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param existing {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.symlink = function(link, existing, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.symlink(link, existing, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param existing {string} 
   @return {FileSystem}
   */
  this.symlinkBlocking = function(link, existing) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.symlinkBlocking(link, existing);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.unlink = function(link, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.unlink(link, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @return {FileSystem}
   */
  this.unlinkBlocking = function(link) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_fileSystem.unlinkBlocking(link);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.readSymlink = function(link, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.readSymlink(link, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param link {string} 
   @return {string}
   */
  this.readSymlinkBlocking = function(link) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_fileSystem.readSymlinkBlocking(link);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.delete = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.delete(path, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileSystem}
   */
  this.deleteBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_fileSystem.deleteBlocking(path);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param recursive {boolean} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.deleteRecursive = function(path, recursive, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] ==='boolean' && typeof __args[2] === 'function') {
      j_fileSystem.deleteRecursive(path, recursive, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param recursive {boolean} 
   @return {FileSystem}
   */
  this.deleteRecursiveBlocking = function(path, recursive) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] ==='boolean') {
      j_fileSystem.deleteRecursiveBlocking(path, recursive);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.mkdir = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.mkdir(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](null, null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.mkdir(__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](null, null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileSystem}
   */
  this.mkdirBlocking = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_fileSystem.mkdirBlocking(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.mkdirBlocking(__args[0], __args[1]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.mkdirs = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.mkdirs(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](null, null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.mkdirs(__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](null, null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileSystem}
   */
  this.mkdirsBlocking = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_fileSystem.mkdirsBlocking(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.mkdirsBlocking(__args[0], __args[1]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.readDir = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.readDir(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](ar.result(), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.readDir(__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](ar.result(), null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {Array.<string>}
   */
  this.readDirBlocking = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_fileSystem.readDirBlocking(__args[0]);
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      return j_fileSystem.readDirBlocking(__args[0], __args[1]);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.readFile = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.readFile(path, function(ar) {
      if (ar.succeeded()) {
        handler(new Buffer(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {Buffer}
   */
  this.readFileBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return new Buffer(j_fileSystem.readFileBlocking(path));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param data {Buffer} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.writeFile = function(path, data, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'function') {
      j_fileSystem.writeFile(path, data._jdel, function(ar) {
      if (ar.succeeded()) {
        handler(null, null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param data {Buffer} 
   @return {FileSystem}
   */
  this.writeFileBlocking = function(path, data) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel) {
      j_fileSystem.writeFileBlocking(path, data._jdel);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param options {Object} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.open = function(path, options, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && typeof __args[2] === 'function') {
      j_fileSystem.open(path, options != null ? new OpenOptions(new JsonObject(JSON.stringify(options))) : null, function(ar) {
      if (ar.succeeded()) {
        handler(new AsyncFile(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param options {Object} 
   @return {AsyncFile}
   */
  this.openBlocking = function(path, options) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object') {
      return new AsyncFile(j_fileSystem.openBlocking(path, options != null ? new OpenOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.createFile = function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.createFile(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](null, null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_fileSystem.createFile(__args[0], __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](null, null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileSystem}
   */
  this.createFileBlocking = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_fileSystem.createFileBlocking(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_fileSystem.createFileBlocking(__args[0], __args[1]);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.exists = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.exists(path, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {boolean}
   */
  this.existsBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_fileSystem.existsBlocking(path);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @param handler {function} 
   @return {FileSystem}
   */
  this.fsProps = function(path, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_fileSystem.fsProps(path, function(ar) {
      if (ar.succeeded()) {
        handler(new FileSystemProps(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param path {string} 
   @return {FileSystemProps}
   */
  this.fsPropsBlocking = function(path) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return new FileSystemProps(j_fileSystem.fsPropsBlocking(path));
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_fileSystem;
};

// We export the Constructor function
module.exports = FileSystem;