/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/http_server */
var utils = require('vertx-js/util/utils');
var HttpServerRequest = require('vertx-js/http_server_request');
var ServerWebSocket = require('vertx-js/server_web_socket');
var Measured = require('vertx-js/measured');
var HttpServerRequestStream = require('vertx-js/http_server_request_stream');
var ServerWebSocketStream = require('vertx-js/server_web_socket_stream');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JHttpServer = io.vertx.core.http.HttpServer;

/**
 An HTTP and WebSockets server<p>
 If an instance is instantiated from an event loop then the handlers
 of the instance will always be called on that same event loop.
 If an instance is instantiated from some other arbitrary Java thread then
 an event loop will be assigned to the instance and used when any of its handlers
 are called.<p>
 Instances of HttpServer are thread-safe.<p>

 @class
*/
var HttpServer = function(j_val) {

  var j_httpServer = j_val;
  var that = this;
  Measured.call(this, j_val);

  /**
   The metric base name

   @public

   @return {string}
   */
  this.metricBaseName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServer.metricBaseName();
    } else utils.invalidArgs();
  };

  /**
   Will return the metrics that correspond with this measured object.

   @public

   @return {Array.<string>}
   */
  this.metrics = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMap(j_httpServer.metrics());
    } else utils.invalidArgs();
  };

  /**
   Return the request stream for the server. As HTTP requests are received by the server,
   @public

   @return {HttpServerRequestStream}
   */
  this.requestStream = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new HttpServerRequestStream(j_httpServer.requestStream());
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param handler {function} 
   @return {HttpServer}
   */
  this.requestHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return new HttpServer(j_httpServer.requestHandler(function(jVal) {
      handler(new HttpServerRequest(jVal));
    }));
    } else utils.invalidArgs();
  };

  /**
   Return the websocket stream for the server. If a websocket connect handshake is successful a
   @public

   @return {ServerWebSocketStream}
   */
  this.websocketStream = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new ServerWebSocketStream(j_httpServer.websocketStream());
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param handler {function} 
   @return {HttpServer}
   */
  this.websocketHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return new HttpServer(j_httpServer.websocketHandler(function(jVal) {
      handler(new ServerWebSocket(jVal));
    }));
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {HttpServer}
   */
  this.listen = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServer.listen();
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServer.listen(function(ar) {
      if (ar.succeeded()) {
        __args[0](new HttpServer(ar.result()), null);
      } else {
        __args[0](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Close the server. Any open HTTP connections will be closed.

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServer.close();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServer.close(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_httpServer;
};

// We export the Constructor function
module.exports = HttpServer;