/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/http_server_request */
var utils = require('vertx-js/util/utils');
var HttpServerFileUpload = require('vertx-js/http_server_file_upload');
var Buffer = require('vertx-js/buffer');
var HttpServerResponse = require('vertx-js/http_server_response');
var ReadStream = require('vertx-js/read_stream');
var MultiMap = require('vertx-js/multi_map');
var SocketAddress = require('vertx-js/socket_address');
var NetSocket = require('vertx-js/net_socket');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JHttpServerRequest = io.vertx.core.http.HttpServerRequest;

/**
 Represents a server-side HTTP request.<p>
 Instances are created for each request that is handled by the server
 @class
*/
var HttpServerRequest = function(j_val) {

  var j_httpServerRequest = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  /**

   @public
   @param handler {function} 
   @return {HttpServerRequest}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerRequest.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpServerRequest}
   */
  this.handler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerRequest.handler(function(jVal) {
      handler(new Buffer(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {HttpServerRequest}
   */
  this.pause = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServerRequest.pause();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {HttpServerRequest}
   */
  this.resume = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServerRequest.resume();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param endHandler {function} 
   @return {HttpServerRequest}
   */
  this.endHandler = function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerRequest.endHandler(endHandler);
      return that;
    } else utils.invalidArgs();
  };

  /**
   The HTTP version of the request

   @public

   @return {Object}
   */
  this.version = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return (j_httpServerRequest.version()).toString();
    } else utils.invalidArgs();
  };

  /**
   The HTTP method for the request. One of GET, PUT, POST, DELETE, TRACE, CONNECT, OPTIONS or HEAD

   @public

   @return {Object}
   */
  this.method = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return (j_httpServerRequest.method()).toString();
    } else utils.invalidArgs();
  };

  /**
   The uri of the request. For example
   http://www.somedomain.com/somepath/somemorepath/someresource.foo?someparam=32&amp;someotherparam=x

   @public

   @return {string}
   */
  this.uri = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerRequest.uri();
    } else utils.invalidArgs();
  };

  /**
   The path part of the uri. For example /somepath/somemorepath/someresource.foo

   @public

   @return {string}
   */
  this.path = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerRequest.path();
    } else utils.invalidArgs();
  };

  /**
   The query part of the uri. For example someparam=32&amp;someotherparam=x

   @public

   @return {string}
   */
  this.query = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerRequest.query();
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {HttpServerResponse}
   */
  this.response = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedresponse == null) {
        that.cachedresponse = new HttpServerResponse(j_httpServerRequest.response());
      }
      return that.cachedresponse;
    } else utils.invalidArgs();
  };

  /**
   A map of all headers in the request, If the request contains multiple headers with the same key, the values
   will be concatenated together into a single header with the same key value, with each value separated by a comma,
   as specified <a href="http://www.w3.org/Protocols/rfc2616/rfc2616-sec4.html#sec4.2">here</a>.
   The headers will be automatically lower-cased when they reach the server

   @public

   @return {MultiMap}
   */
  this.headers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedheaders == null) {
        that.cachedheaders = new MultiMap(j_httpServerRequest.headers());
      }
      return that.cachedheaders;
    } else utils.invalidArgs();
  };

  /**
   Returns a map of all the parameters in the request

   @public

   @return {MultiMap}
   */
  this.params = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedparams == null) {
        that.cachedparams = new MultiMap(j_httpServerRequest.params());
      }
      return that.cachedparams;
    } else utils.invalidArgs();
  };

  /**
   Return the remote (client side) address of the request

   @public

   @return {SocketAddress}
   */
  this.remoteAddress = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedremoteAddress == null) {
        that.cachedremoteAddress = new SocketAddress(j_httpServerRequest.remoteAddress());
      }
      return that.cachedremoteAddress;
    } else utils.invalidArgs();
  };

  /**
   Return the local (server side) address of the server that handles the request

   @public

   @return {SocketAddress}
   */
  this.localAddress = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedlocalAddress == null) {
        that.cachedlocalAddress = new SocketAddress(j_httpServerRequest.localAddress());
      }
      return that.cachedlocalAddress;
    } else utils.invalidArgs();
  };

  /**
   Get the absolute URI corresponding to the the HTTP request
   @public

   @return {string}
   */
  this.absoluteURI = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerRequest.absoluteURI();
    } else utils.invalidArgs();
  };

  /**
   Convenience method for receiving the entire request body in one piece. This saves the user having to manually
   set a data and end handler and append the chunks of the body until the whole body received.
   Don't use this if your request body is large - you could potentially run out of RAM.

   @public
   @param bodyHandler {function} 
   @return {HttpServerRequest}
   */
  this.bodyHandler = function(bodyHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerRequest.bodyHandler(function(jVal) {
      bodyHandler(new Buffer(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Get a net socket for the underlying connection of this request. USE THIS WITH CAUTION!
   Writing to the socket directly if you don't know what you're doing can easily break the HTTP protocol
   @public

   @return {NetSocket}
   */
  this.netSocket = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachednetSocket == null) {
        that.cachednetSocket = new NetSocket(j_httpServerRequest.netSocket());
      }
      return that.cachednetSocket;
    } else utils.invalidArgs();
  };

  /**
   Call this with true if you are expecting a multi-part form to be submitted in the request
   This must be called before the body of the request has been received
   @public
   @param expect {boolean} 
   @return {HttpServerRequest}
   */
  this.setExpectMultipart = function(expect) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_httpServerRequest.setExpectMultipart(expect);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {boolean}
   */
  this.isExpectMultipart = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerRequest.isExpectMultipart();
    } else utils.invalidArgs();
  };

  /**
   Set the upload handler. The handler will get notified once a new file upload was received and so allow to
   get notified by the upload in progress.

   @public
   @param uploadHandler {function} 
   @return {HttpServerRequest}
   */
  this.uploadHandler = function(uploadHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerRequest.uploadHandler(function(jVal) {
      uploadHandler(new HttpServerFileUpload(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Returns a map of all form attributes which was found in the request. Be aware that this message should only get
   called after the endHandler was notified as the map will be filled on-the-fly.
   @public

   @return {MultiMap}
   */
  this.formAttributes = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedformAttributes == null) {
        that.cachedformAttributes = new MultiMap(j_httpServerRequest.formAttributes());
      }
      return that.cachedformAttributes;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_httpServerRequest;
};

// We export the Constructor function
module.exports = HttpServerRequest;