/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/http_server_response */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');
var WriteStream = require('vertx-js/write_stream');
var MultiMap = require('vertx-js/multi_map');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JHttpServerResponse = io.vertx.core.http.HttpServerResponse;

/**
 Represents a server-side HTTP response.<p>
 Instances of this class are created and associated to every instance of
 @class
*/
var HttpServerResponse = function(j_val) {

  var j_httpServerResponse = j_val;
  var that = this;
  WriteStream.call(this, j_val);

  /**
  
   @public

   @return {boolean}
   */
  this.writeQueueFull = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.writeQueueFull();
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpServerResponse}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerResponse.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param data {Buffer} 
   @return {HttpServerResponse}
   */
  this.write = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_httpServerResponse.write(__args[0]._jdel);
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpServerResponse.write(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpServerResponse.write(__args[0], __args[1]);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param maxSize {number} 
   @return {HttpServerResponse}
   */
  this.setWriteQueueMaxSize = function(maxSize) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_httpServerResponse.setWriteQueueMaxSize(maxSize);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpServerResponse}
   */
  this.drainHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerResponse.drainHandler(handler);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {number}
   */
  this.getStatusCode = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.getStatusCode();
    } else utils.invalidArgs();
  };

  /**
   Set the status code
   @public
   @param statusCode {number} 
   @return {HttpServerResponse}
   */
  this.setStatusCode = function(statusCode) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_httpServerResponse.setStatusCode(statusCode);
      return that;
    } else utils.invalidArgs();
  };

  /**
   The HTTP status message of the response. If this is not specified a default value will be used depending on what
   @public

   @return {string}
   */
  this.getStatusMessage = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.getStatusMessage();
    } else utils.invalidArgs();
  };

  /**
   Set the status message
   @public
   @param statusMessage {string} 
   @return {HttpServerResponse}
   */
  this.setStatusMessage = function(statusMessage) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpServerResponse.setStatusMessage(statusMessage);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param chunked {boolean} 
   @return {HttpServerResponse}
   */
  this.setChunked = function(chunked) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_httpServerResponse.setChunked(chunked);
      return that;
    } else utils.invalidArgs();
  };

  /**
   Is the response chunked?

   @public

   @return {boolean}
   */
  this.isChunked = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.isChunked();
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {MultiMap}
   */
  this.headers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedheaders == null) {
        that.cachedheaders = new MultiMap(j_httpServerResponse.headers());
      }
      return that.cachedheaders;
    } else utils.invalidArgs();
  };

  /**
   Put an HTTP header - fluent API
   @public
   @param name {string} 
   @param value {string} 
   @return {HttpServerResponse}
   */
  this.putHeader = function(name, value) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpServerResponse.putHeader(name, value);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {MultiMap}
   */
  this.trailers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedtrailers == null) {
        that.cachedtrailers = new MultiMap(j_httpServerResponse.trailers());
      }
      return that.cachedtrailers;
    } else utils.invalidArgs();
  };

  /**
   Put an HTTP trailer - fluent API
   @public
   @param name {string} 
   @param value {string} 
   @return {HttpServerResponse}
   */
  this.putTrailer = function(name, value) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpServerResponse.putTrailer(name, value);
      return that;
    } else utils.invalidArgs();
  };

  /**
   Set a close handler for the response. This will be called if the underlying connection closes before the response
   is complete.
   @public
   @param handler {function} 
   @return {HttpServerResponse}
   */
  this.closeHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerResponse.closeHandler(handler);
      return that;
    } else utils.invalidArgs();
  };

  /**
   Ends the response. If no data has been written to the response body,
   the actual response won't get written until this method gets called.<p>
   Once the response has ended, it cannot be used any more.

   @public

   */
  this.end = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServerResponse.end();
    }  else if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpServerResponse.end(__args[0]);
    }  else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_httpServerResponse.end(__args[0]._jdel);
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpServerResponse.end(__args[0], __args[1]);
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param filename {string} 
   @return {HttpServerResponse}
   */
  this.sendFile = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpServerResponse.sendFile(__args[0]);
      return that;
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_httpServerResponse.sendFile(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](null, null);
      } else {
        __args[1](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Close the underlying TCP connection

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpServerResponse.close();
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {boolean}
   */
  this.ended = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.ended();
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {boolean}
   */
  this.headWritten = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpServerResponse.headWritten();
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpServerResponse}
   */
  this.headersEndHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerResponse.headersEndHandler(handler);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpServerResponse}
   */
  this.bodyEndHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpServerResponse.bodyEndHandler(handler);
      return that;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_httpServerResponse;
};

// We export the Constructor function
module.exports = HttpServerResponse;