/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/async_file */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JAsyncFile = io.vertx.core.file.AsyncFile;

/**
 Represents a file on the file-system which can be read from, or written to asynchronously.<p>
 @class
*/
var AsyncFile = function(j_val) {

  var j_asyncFile = j_val;
  var that = this;
  ReadStream.call(this, j_val);
  WriteStream.call(this, j_val);

  /**
  
   @public

   @return {boolean}
   */
  this.writeQueueFull = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_asyncFile.writeQueueFull();
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {AsyncFile}
   */
  this.handler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.handler(function(jVal) {
      handler(new Buffer(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {AsyncFile}
   */
  this.pause = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_asyncFile.pause();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {AsyncFile}
   */
  this.resume = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_asyncFile.resume();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param endHandler {function} 
   @return {AsyncFile}
   */
  this.endHandler = function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.endHandler(endHandler);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param data {Buffer} 
   @return {AsyncFile}
   */
  this.write = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_asyncFile.write(__args[0]._jdel);
      return that;
    }  else if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] ==='number' && typeof __args[2] === 'function') {
      j_asyncFile.write(__args[0]._jdel, __args[1], function(ar) {
      if (ar.succeeded()) {
        __args[2](null, null);
      } else {
        __args[2](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param maxSize {number} 
   @return {AsyncFile}
   */
  this.setWriteQueueMaxSize = function(maxSize) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_asyncFile.setWriteQueueMaxSize(maxSize);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {AsyncFile}
   */
  this.drainHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.drainHandler(handler);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {AsyncFile}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Close the file. The actual close happens asynchronously.

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_asyncFile.close();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.close(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param buffer {Buffer} 
   @param offset {number} 
   @param position {number} 
   @param length {number} 
   @param handler {function} 
   @return {AsyncFile}
   */
  this.read = function(buffer, offset, position, length, handler) {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] ==='number' && typeof __args[4] === 'function') {
      j_asyncFile.read(buffer._jdel, offset, position, length, function(ar) {
      if (ar.succeeded()) {
        handler(new Buffer(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Flush any writes made to this file to underlying persistent storage.<p>
   @public

   @return {AsyncFile}
   */
  this.flush = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_asyncFile.flush();
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_asyncFile.flush(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param readPos {number} 
   @return {AsyncFile}
   */
  this.setReadPos = function(readPos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_asyncFile.setReadPos(readPos);
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param readPos {number} 
   @return {AsyncFile}
   */
  this.setWritePos = function(readPos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_asyncFile.setWritePos(readPos);
      return that;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_asyncFile;
};

// We export the Constructor function
module.exports = AsyncFile;