/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/dns_client */
var utils = require('vertx-js/util/utils');
var MxRecord = require('vertx-js/mx_record');
var SrvRecord = require('vertx-js/srv_record');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JDnsClient = io.vertx.core.dns.DnsClient;

/**
 Provides a way to asynchronous lookup informations from DNS-Servers.

 @class
*/
var DnsClient = function(j_val) {

  var j_dnsClient = j_val;
  var that = this;

  /**
   Try to lookup the A (ipv4) or AAAA (ipv6) record for the given name. The first found will be used.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.lookup = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.lookup(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to lookup the A (ipv4) record for the given name. The first found will be used.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.lookup4 = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.lookup4(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to lookup the AAAA (ipv6) record for the given name. The first found will be used.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.lookup6 = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.lookup6(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve all A (ipv4) records for the given name.
  
  

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveA = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveA(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve all AAAA (ipv6) records for the given name.
  

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveAAAA = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveAAAA(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the CNAME record for the given name.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveCNAME = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveCNAME(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the MX records for the given name.
  

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveMX = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveMX(name, function(ar) {
      if (ar.succeeded()) {
        handler(utils.convReturnListSetVertxGen(ar.result(), MxRecord), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the TXT records for the given name.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveTXT = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveTXT(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the PTR record for the given name.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolvePTR = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolvePTR(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the NS records for the given name.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveNS = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveNS(name, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to resolve the SRV records for the given name.

   @public
   @param name {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.resolveSRV = function(name, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.resolveSRV(name, function(ar) {
      if (ar.succeeded()) {
        handler(utils.convReturnListSetVertxGen(ar.result(), SrvRecord), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Try to do a reverse lookup of an ipaddress. This is basically the same as doing trying to resolve a PTR record
   but allows you to just pass in the ipaddress and not a valid ptr query string.

   @public
   @param ipaddress {string} 
   @param handler {function} 
   @return {DnsClient}
   */
  this.reverseLookup = function(ipaddress, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_dnsClient.reverseLookup(ipaddress, function(ar) {
      if (ar.succeeded()) {
        handler(ar.result(), null);
      } else {
        handler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_dnsClient;
};

// We export the Constructor function
module.exports = DnsClient;