/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/future */
var utils = require('vertx-js/util/utils');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JFuture = io.vertx.core.Future;

/**
 @class
*/
var Future = function(j_val) {

  var j_future = j_val;
  var that = this;

  /**
   Has it completed?

   @public

   @return {boolean}
   */
  this.isComplete = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_future.isComplete();
    } else utils.invalidArgs();
  };

  /**
   Set a handler for the result. It will get called when it's complete

   @public
   @param handler {function} 
   */
  this.setHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_future.setHandler(function(ar) {
      if (ar.succeeded()) {
        handler(utils.convReturnTypeUnknown(ar.result()), null);
      } else {
        handler(null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**

   @public

   */
  this.complete = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_future.complete();
    }  else if (__args.length === 1 && true) {
      j_future.complete(utils.convParamTypeUnknown(__args[0]));
    } else utils.invalidArgs();
  };

  /**

   @public
   @param failureMessage {string} 
   */
  this.fail = function(failureMessage) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_future.fail(failureMessage);
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_future;
};

/**

 @memberof module:vertx-js/future

 @return {Future}
 */
Future.future = function() {
  var __args = arguments;
  if (__args.length === 0) {
    return new Future(JFuture.future());
  } else utils.invalidArgs();
};

/**

 @memberof module:vertx-js/future

 @return {Future}
 */
Future.succeededFuture = function() {
  var __args = arguments;
  if (__args.length === 0) {
    return new Future(JFuture.succeededFuture());
  }else if (__args.length === 1 && true) {
    return new Future(JFuture.succeededFuture(utils.convParamTypeUnknown(__args[0])));
  } else utils.invalidArgs();
};

/**

 @memberof module:vertx-js/future
 @param failureMessage {string} 
 @return {Future}
 */
Future.failedFuture = function(failureMessage) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'string') {
    return new Future(JFuture.failedFuture(failureMessage));
  } else utils.invalidArgs();
};

// We export the Constructor function
module.exports = Future;