/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/http_client_response */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');
var ReadStream = require('vertx-js/read_stream');
var MultiMap = require('vertx-js/multi_map');
var NetSocket = require('vertx-js/net_socket');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JHttpClientResponse = io.vertx.core.http.HttpClientResponse;

/**
 Represents a client-side HTTP response.<p>
 @class
*/
var HttpClientResponse = function(j_val) {

  var j_httpClientResponse = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  /**

   @public

   @return {HttpClientResponse}
   */
  this.resume = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpClientResponse.resume();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpClientResponse}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpClientResponse.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {HttpClientResponse}
   */
  this.handler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpClientResponse.handler(function(jVal) {
      handler(new Buffer(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {HttpClientResponse}
   */
  this.pause = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_httpClientResponse.pause();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param endHandler {function} 
   @return {HttpClientResponse}
   */
  this.endHandler = function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpClientResponse.endHandler(endHandler);
      return that;
    } else utils.invalidArgs();
  };

  /**
   The HTTP status code of the response

   @public

   @return {number}
   */
  this.statusCode = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpClientResponse.statusCode();
    } else utils.invalidArgs();
  };

  /**
   The HTTP status message of the response

   @public

   @return {string}
   */
  this.statusMessage = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_httpClientResponse.statusMessage();
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {MultiMap}
   */
  this.headers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedheaders == null) {
        that.cachedheaders = new MultiMap(j_httpClientResponse.headers());
      }
      return that.cachedheaders;
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {MultiMap}
   */
  this.trailers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedtrailers == null) {
        that.cachedtrailers = new MultiMap(j_httpClientResponse.trailers());
      }
      return that.cachedtrailers;
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {Array.<string>}
   */
  this.cookies = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedcookies == null) {
        that.cachedcookies = j_httpClientResponse.cookies();
      }
      return that.cachedcookies;
    } else utils.invalidArgs();
  };

  /**
   Convenience method for receiving the entire request body in one piece. This saves the user having to manually
   set a data and end handler and append the chunks of the body until the whole body received.
   Don't use this if your request body is large - you could potentially run out of RAM.

   @public
   @param bodyHandler {function} 
   @return {HttpClientResponse}
   */
  this.bodyHandler = function(bodyHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpClientResponse.bodyHandler(function(jVal) {
      bodyHandler(new Buffer(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Get a net socket for the underlying connection of this request. USE THIS WITH CAUTION!
   Writing to the socket directly if you don't know what you're doing can easily break the HTTP protocol

   @public

   @return {NetSocket}
   */
  this.netSocket = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachednetSocket == null) {
        that.cachednetSocket = new NetSocket(j_httpClientResponse.netSocket());
      }
      return that.cachednetSocket;
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_httpClientResponse;
};

// We export the Constructor function
module.exports = HttpClientResponse;