/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/message */
var utils = require('vertx-js/util/utils');
var MultiMap = require('vertx-js/multi_map');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JMessage = io.vertx.core.eventbus.Message;
var DeliveryOptions = io.vertx.core.eventbus.DeliveryOptions;
var DeliveryOptions = io.vertx.core.eventbus.DeliveryOptions;

/**

 @class
*/
var Message = function(j_val) {

  var j_message = j_val;
  var that = this;

  /**
   The address the message was sent to

   @public

   @return {string}
   */
  this.address = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_message.address();
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {MultiMap}
   */
  this.headers = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new MultiMap(j_message.headers());
    } else utils.invalidArgs();
  };

  /**
   The body of the message

   @public

   @return {Object}
   */
  this.body = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedbody == null) {
        that.cachedbody = utils.convReturnTypeUnknown(j_message.body());
      }
      return that.cachedbody;
    } else utils.invalidArgs();
  };

  /**
   The reply address (if any)

   @public

   @return {string}
   */
  this.replyAddress = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_message.replyAddress();
    } else utils.invalidArgs();
  };

  /**
   Reply to this message. If the message was sent specifying a reply handler, that handler will be
   called when it has received a reply. If the message wasn't sent specifying a receipt handler
   this method does nothing.

   @public
   @param message {Object} 
   */
  this.reply = function() {
    var __args = arguments;
    if (__args.length === 1 && true) {
      j_message.reply(utils.convParamTypeUnknown(__args[0]));
    }  else if (__args.length === 2 && true && typeof __args[1] === 'function') {
      j_message.reply(utils.convParamTypeUnknown(__args[0]), function(ar) {
      if (ar.succeeded()) {
        __args[1](new Message(ar.result()), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
    }  else if (__args.length === 2 && true && typeof __args[1] === 'object') {
      j_message.reply(utils.convParamTypeUnknown(__args[0]), __args[1] != null ? new DeliveryOptions(new JsonObject(JSON.stringify(__args[1]))) : null);
    }  else if (__args.length === 3 && true && typeof __args[1] === 'object' && typeof __args[2] === 'function') {
      j_message.reply(utils.convParamTypeUnknown(__args[0]), __args[1] != null ? new DeliveryOptions(new JsonObject(JSON.stringify(__args[1]))) : null, function(ar) {
      if (ar.succeeded()) {
        __args[2](new Message(ar.result()), null);
      } else {
        __args[2](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**
   Signal that processing of this message failed. If the message was sent specifying a result handler
   the handler will be called with a failure corresponding to the failure code and message specified here
   @public
   @param failureCode {number} 
   @param message {string} 
   */
  this.fail = function(failureCode, message) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'string') {
      j_message.fail(failureCode, message);
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_message;
};

// We export the Constructor function
module.exports = Message;