/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/message_consumer */
var utils = require('vertx-js/util/utils');
var ReadStream = require('vertx-js/read_stream');
var Message = require('vertx-js/message');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JMessageConsumer = io.vertx.core.eventbus.MessageConsumer;

/**

 @class
*/
var MessageConsumer = function(j_val) {

  var j_messageConsumer = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  /**

   @public
   @param handler {function} 
   @return {MessageConsumer}
   */
  this.exceptionHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_messageConsumer.exceptionHandler(function(jVal) {
      handler(utils.convReturnTypeUnknown(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param handler {function} 
   @return {MessageConsumer}
   */
  this.handler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_messageConsumer.handler(function(jVal) {
      handler(new Message(jVal));
    });
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {MessageConsumer}
   */
  this.pause = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_messageConsumer.pause();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {MessageConsumer}
   */
  this.resume = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_messageConsumer.resume();
      return that;
    } else utils.invalidArgs();
  };

  /**

   @public
   @param endHandler {function} 
   @return {MessageConsumer}
   */
  this.endHandler = function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_messageConsumer.endHandler(endHandler);
      return that;
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {ReadStream}
   */
  this.bodyStream = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new ReadStream(j_messageConsumer.bodyStream());
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {boolean}
   */
  this.isRegistered = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_messageConsumer.isRegistered();
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {string}
   */
  this.address = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_messageConsumer.address();
    } else utils.invalidArgs();
  };

  /**
   Set the number of messages this registration will buffer when this stream is paused. The default
   value is <code>0</code>. When a new value is set, buffered messages may be discarded to reach
   the new value.

   @public
   @param maxBufferedMessages {number} 
   @return {MessageConsumer}
   */
  this.setMaxBufferedMessages = function(maxBufferedMessages) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return new MessageConsumer(j_messageConsumer.setMaxBufferedMessages(maxBufferedMessages));
    } else utils.invalidArgs();
  };

  /**
  
   @public

   @return {number}
   */
  this.getMaxBufferedMessages = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_messageConsumer.getMaxBufferedMessages();
    } else utils.invalidArgs();
  };

  /**
   Optional method which can be called to indicate when the registration has been propagated across the cluster.

   @public
   @param completionHandler {function} 
   */
  this.completionHandler = function(completionHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_messageConsumer.completionHandler(function(ar) {
      if (ar.succeeded()) {
        completionHandler(null, null);
      } else {
        completionHandler(null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**
   Unregisters the handler which created this registration

   @public

   */
  this.unregister = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_messageConsumer.unregister();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_messageConsumer.unregister(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_messageConsumer;
};

// We export the Constructor function
module.exports = MessageConsumer;