/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/net_client */
var utils = require('vertx-js/util/utils');
var Measured = require('vertx-js/measured');
var NetSocket = require('vertx-js/net_socket');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JNetClient = io.vertx.core.net.NetClient;

/**
 A TCP/SSL client.<p>
 Multiple connections to different servers can be made using the same instance.<p>
 This client supports a configurable number of connection attempts and a configurable
 delay between attempts.<p>
 If an instance is instantiated from an event loop then the handlers
 of the instance will always be called on that same event loop.
 If an instance is instantiated from some other arbitrary Java thread (i.e. when using embedded) then
 an event loop will be assigned to the instance and used when any of its handlers
 are called.<p>
 Instances of this class are thread-safe.<p>

 @class
*/
var NetClient = function(j_val) {

  var j_netClient = j_val;
  var that = this;
  Measured.call(this, j_val);

  /**
   The metric base name

   @public

   @return {string}
   */
  this.metricBaseName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_netClient.metricBaseName();
    } else utils.invalidArgs();
  };

  /**
   Will return the metrics that correspond with this measured object.

   @public

   @return {Array.<string>}
   */
  this.metrics = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMap(j_netClient.metrics());
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param port {number} 
   @param host {string} 
   @param connectHandler {function} 
   @return {NetClient}
   */
  this.connect = function(port, host, connectHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] ==='number' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_netClient.connect(port, host, function(ar) {
      if (ar.succeeded()) {
        connectHandler(new NetSocket(ar.result()), null);
      } else {
        connectHandler(null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Close the client. Any sockets which have not been closed manually will be closed here.

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_netClient.close();
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_netClient;
};

// We export the Constructor function
module.exports = NetClient;