/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/net_server */
var utils = require('vertx-js/util/utils');
var NetSocketStream = require('vertx-js/net_socket_stream');
var Measured = require('vertx-js/measured');
var NetSocket = require('vertx-js/net_socket');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JNetServer = io.vertx.core.net.NetServer;

/**
 Represents a TCP or SSL server<p>
 If an instance is instantiated from an event loop then the handlers
 of the instance will always be called on that same event loop.
 If an instance is instantiated from some other arbitrary Java thread (i.e. when running embedded) then
 and event loop will be assigned to the instance and used when any of its handlers
 are called.<p>
 Instances of this class are thread-safe.<p>

 @class
*/
var NetServer = function(j_val) {

  var j_netServer = j_val;
  var that = this;
  Measured.call(this, j_val);

  /**
   The metric base name

   @public

   @return {string}
   */
  this.metricBaseName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_netServer.metricBaseName();
    } else utils.invalidArgs();
  };

  /**
   Will return the metrics that correspond with this measured object.

   @public

   @return {Array.<string>}
   */
  this.metrics = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMap(j_netServer.metrics());
    } else utils.invalidArgs();
  };

  /**
   Return the connect stream for this server. The server can only have at most one handler at any one time.
   @public

   @return {NetSocketStream}
   */
  this.connectStream = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new NetSocketStream(j_netServer.connectStream());
    } else utils.invalidArgs();
  };

  /**
   Supply a connect handler for this server. The server can only have at most one connect handler at any one time.
   @public
   @param handler {function} 
   @return {NetServer}
   */
  this.connectHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return new NetServer(j_netServer.connectHandler(function(jVal) {
      handler(new NetSocket(jVal));
    }));
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {NetServer}
   */
  this.listen = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_netServer.listen();
      return that;
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_netServer.listen(function(ar) {
      if (ar.succeeded()) {
        __args[0](new NetServer(ar.result()), null);
      } else {
        __args[0](null, ar.cause());
      }
    });
      return that;
    } else utils.invalidArgs();
  };

  /**
   Close the server. This will close any currently open connections.

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_netServer.close();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_netServer.close(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**
   The actual port the server is listening on. This is useful if you bound the server specifying 0 as port number
   signifying an ephemeral port

   @public

   @return {number}
   */
  this.actualPort = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_netServer.actualPort();
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_netServer;
};

// We export the Constructor function
module.exports = NetServer;