/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/vertx */
var utils = require('vertx-js/util/utils');
var DatagramSocket = require('vertx-js/datagram_socket');
var HttpServer = require('vertx-js/http_server');
var Context = require('vertx-js/context');
var Future = require('vertx-js/future');
var SharedData = require('vertx-js/shared_data');
var DnsClient = require('vertx-js/dns_client');
var TimeoutStream = require('vertx-js/timeout_stream');
var EventBus = require('vertx-js/event_bus');
var Measured = require('vertx-js/measured');
var NetServer = require('vertx-js/net_server');
var FileSystem = require('vertx-js/file_system');
var NetClient = require('vertx-js/net_client');
var HttpClient = require('vertx-js/http_client');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JVertx = io.vertx.core.Vertx;
var VertxOptions = io.vertx.core.VertxOptions;
var VertxOptions = io.vertx.core.VertxOptions;
var NetServerOptions = io.vertx.core.net.NetServerOptions;
var NetClientOptions = io.vertx.core.net.NetClientOptions;
var HttpServerOptions = io.vertx.core.http.HttpServerOptions;
var HttpClientOptions = io.vertx.core.http.HttpClientOptions;
var DatagramSocketOptions = io.vertx.core.datagram.DatagramSocketOptions;
var DeploymentOptions = io.vertx.core.DeploymentOptions;
var DeploymentOptions = io.vertx.core.DeploymentOptions;

/**
 The control centre of the Vert.x Core API.<p>
 You should normally only use a single instance of this class throughout your application. If you are running in the
 Vert.x container an instance will be provided to you.<p>
 @class
*/
var Vertx = function(j_val) {

  var j_vertx = j_val;
  var that = this;
  Measured.call(this, j_val);

  /**
   The metric base name

   @public

   @return {string}
   */
  this.metricBaseName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_vertx.metricBaseName();
    } else utils.invalidArgs();
  };

  /**
   Will return the metrics that correspond with this measured object.

   @public

   @return {Array.<string>}
   */
  this.metrics = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMap(j_vertx.metrics());
    } else utils.invalidArgs();
  };

  /**
   Gets the current context, or creates one if there isn't one
   @public

   @return {Context}
   */
  this.getOrCreateContext = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return new Context(j_vertx.getOrCreateContext());
    } else utils.invalidArgs();
  };

  /**
   Create a TCP/SSL server

   @public
   @param options {Object} 
   @return {NetServer}
   */
  this.createNetServer = function(options) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object') {
      return new NetServer(j_vertx.createNetServer(options != null ? new NetServerOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**
   Create a TCP/SSL client

   @public
   @param options {Object} 
   @return {NetClient}
   */
  this.createNetClient = function(options) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object') {
      return new NetClient(j_vertx.createNetClient(options != null ? new NetClientOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**
   Create an HTTP/HTTPS server

   @public
   @param options {Object} 
   @return {HttpServer}
   */
  this.createHttpServer = function(options) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object') {
      return new HttpServer(j_vertx.createHttpServer(options != null ? new HttpServerOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**
   Create a HTTP/HTTPS client

   @public
   @param options {Object} 
   @return {HttpClient}
   */
  this.createHttpClient = function(options) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object') {
      return new HttpClient(j_vertx.createHttpClient(options != null ? new HttpClientOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**

   @public
   @param options {Object} 
   @return {DatagramSocket}
   */
  this.createDatagramSocket = function(options) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object') {
      return new DatagramSocket(j_vertx.createDatagramSocket(options != null ? new DatagramSocketOptions(new JsonObject(JSON.stringify(options))) : null));
    } else utils.invalidArgs();
  };

  /**
   The File system object

   @public

   @return {FileSystem}
   */
  this.fileSystem = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedfileSystem == null) {
        that.cachedfileSystem = new FileSystem(j_vertx.fileSystem());
      }
      return that.cachedfileSystem;
    } else utils.invalidArgs();
  };

  /**
   The event bus

   @public

   @return {EventBus}
   */
  this.eventBus = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedeventBus == null) {
        that.cachedeventBus = new EventBus(j_vertx.eventBus());
      }
      return that.cachedeventBus;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param port {number} 
   @param host {string} 
   @return {DnsClient}
   */
  this.createDnsClient = function(port, host) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'string') {
      return new DnsClient(j_vertx.createDnsClient(port, host));
    } else utils.invalidArgs();
  };

  /**
   The shared data object

   @public

   @return {SharedData}
   */
  this.sharedData = function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedsharedData == null) {
        that.cachedsharedData = new SharedData(j_vertx.sharedData());
      }
      return that.cachedsharedData;
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param delay {number} 
   @param handler {function} 
   @return {number}
   */
  this.setTimer = function(delay, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'function') {
      return j_vertx.setTimer(delay, function(jVal) {
      handler(jVal);
    });
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param delay {number} 
   @return {TimeoutStream}
   */
  this.timerStream = function(delay) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return new TimeoutStream(j_vertx.timerStream(delay));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param delay {number} 
   @param handler {function} 
   @return {number}
   */
  this.setPeriodic = function(delay, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'function') {
      return j_vertx.setPeriodic(delay, function(jVal) {
      handler(jVal);
    });
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param delay {number} 
   @return {TimeoutStream}
   */
  this.periodicStream = function(delay) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return new TimeoutStream(j_vertx.periodicStream(delay));
    } else utils.invalidArgs();
  };

  /**
  
   @public
   @param id {number} 
   @return {boolean}
   */
  this.cancelTimer = function(id) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_vertx.cancelTimer(id);
    } else utils.invalidArgs();
  };

  /**
   Put the handler on the event queue for the current loop (or worker context) so it will be run asynchronously ASAP after this event has
   been processed

   @public
   @param action {function} 
   */
  this.runOnContext = function(action) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_vertx.runOnContext(action);
    } else utils.invalidArgs();
  };

  /**
   Stop the eventbus and any resource managed by the eventbus.

   @public

   */
  this.close = function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_vertx.close();
    }  else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_vertx.close(function(ar) {
      if (ar.succeeded()) {
        __args[0](null, null);
      } else {
        __args[0](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**

   @public
   @param identifier {string} 
   */
  this.deployVerticle = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_vertx.deployVerticle(__args[0]);
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_vertx.deployVerticle(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](ar.result(), null);
      } else {
        __args[1](null, ar.cause());
      }
    });
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object') {
      j_vertx.deployVerticle(__args[0], __args[1] != null ? new DeploymentOptions(new JsonObject(JSON.stringify(__args[1]))) : null);
    }  else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && typeof __args[2] === 'function') {
      j_vertx.deployVerticle(__args[0], __args[1] != null ? new DeploymentOptions(new JsonObject(JSON.stringify(__args[1]))) : null, function(ar) {
      if (ar.succeeded()) {
        __args[2](ar.result(), null);
      } else {
        __args[2](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**

   @public
   @param deploymentID {string} 
   */
  this.undeployVerticle = function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_vertx.undeployVerticle(__args[0]);
    }  else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_vertx.undeployVerticle(__args[0], function(ar) {
      if (ar.succeeded()) {
        __args[1](null, null);
      } else {
        __args[1](null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  /**

   @public

   @return {Array.<string>}
   */
  this.deployments = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnSet(j_vertx.deployments());
    } else utils.invalidArgs();
  };

  /**

   @public
   @param blockingCodeHandler {function} 
   @param resultHandler {function} 
   */
  this.executeBlocking = function(blockingCodeHandler, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'function' && typeof __args[1] === 'function') {
      j_vertx.executeBlocking(function(jVal) {
      blockingCodeHandler(new Future(jVal));
    }, function(ar) {
      if (ar.succeeded()) {
        resultHandler(utils.convReturnTypeUnknown(ar.result()), null);
      } else {
        resultHandler(null, ar.cause());
      }
    });
    } else utils.invalidArgs();
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_vertx;
};

/**

 @memberof module:vertx-js/vertx

 @return {Vertx}
 */
Vertx.vertx = function() {
  var __args = arguments;
  if (__args.length === 0) {
    return new Vertx(JVertx.vertx());
  }else if (__args.length === 1 && typeof __args[0] === 'object') {
    return new Vertx(JVertx.vertx(__args[0] != null ? new VertxOptions(new JsonObject(JSON.stringify(__args[0]))) : null));
  } else utils.invalidArgs();
};

/**

 @memberof module:vertx-js/vertx
 @param options {Object} 
 @param resultHandler {function} 
 */
Vertx.clusteredVertx = function(options, resultHandler) {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && typeof __args[1] === 'function') {
    JVertx.clusteredVertx(options != null ? new VertxOptions(new JsonObject(JSON.stringify(options))) : null, function(ar) {
    if (ar.succeeded()) {
      resultHandler(new Vertx(ar.result()), null);
    } else {
      resultHandler(null, ar.cause());
    }
  });
  } else utils.invalidArgs();
};

/**
 Gets the current context
 @memberof module:vertx-js/vertx

 @return {Context}
 */
Vertx.currentContext = function() {
  var __args = arguments;
  if (__args.length === 0) {
    return new Context(JVertx.currentContext());
  } else utils.invalidArgs();
};

// We export the Constructor function
module.exports = Vertx;