/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-common-js/template_engine */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JTemplateEngine = Java.type('io.vertx.ext.web.common.template.TemplateEngine');

/**
 A template template uses a specific template and the data in a routing context to render a resource into a buffer.
 <p>
 Concrete implementations exist for several well-known template engines.

 @class
*/
var TemplateEngine = function(j_val) {

  var j_templateEngine = j_val;
  var that = this;

  /**
   Render the template. Template engines that support partials/fragments should extract the template base path from
   the template filename up to the last file separator.

   Some engines support localization, for these engines, there is a predefined key "lang" to specify the language to
   be used in the localization, the format should follow the standard locale formats e.g.: "en-gb", "pt-br", "en".

   @public
   @param context {Object} the routing context 
   @param templateFileName {string} the template file name to use 
   @param handler {function} the handler that will be called with a result containing the buffer or a failure. 
   */
  this.render =  function(context, templateFileName, handler) {
    var __args = arguments;
    if (__args.length === 3 && (typeof __args[0] === 'object' && __args[0] != null) && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_templateEngine["render(io.vertx.core.json.JsonObject,java.lang.String,io.vertx.core.Handler)"](utils.convParamJsonObject(context), templateFileName, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(Buffer, ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns true if the template template caches template files. If false, then template files are freshly loaded each
   time they are used.

   @public

   @return {boolean} True if template files are cached; otherwise, false.
   */
  this.isCachingEnabled =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_templateEngine["isCachingEnabled()"]() ;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_templateEngine;
};

TemplateEngine._jclass = utils.getJavaClass("io.vertx.ext.web.common.template.TemplateEngine");
TemplateEngine._jtype = {accept: function(obj) {
    return TemplateEngine._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(TemplateEngine.prototype, {});
    TemplateEngine.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
TemplateEngine._create = function(jdel) {var obj = Object.create(TemplateEngine.prototype, {});
  TemplateEngine.apply(obj, arguments);
  return obj;
}
module.exports = TemplateEngine;