/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-rabbitmq-js/rabbit_mq_consumer */
var utils = require('vertx-js/util/utils');
var RabbitMQMessage = require('vertx-rabbitmq-js/rabbit_mq_message');
var ReadStream = require('vertx-js/read_stream');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JRabbitMQConsumer = Java.type('io.vertx.rabbitmq.RabbitMQConsumer');

/**
 A stream of messages from a rabbitmq queue.

 @class
*/
var RabbitMQConsumer = function(j_val) {

  var j_rabbitMQConsumer = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_fetch = this.fetch;
  var __super_exceptionHandler = this.exceptionHandler;
  var __super_handler = this.handler;
  var __super_pause = this.pause;
  var __super_resume = this.resume;
  var __super_endHandler = this.endHandler;
  var __super_consumerTag = this.consumerTag;
  var __super_cancel = this.cancel;
  var __super_cancel = this.cancel;
  var __super_isPaused = this.isPaused;
  /**
   Fetch the specified <code>amount</code> of elements. If the <code>ReadStream</code> has been paused, reading will
   recommence with the specified <code>amount</code> of items, otherwise the specified <code>amount</code> will
   be added to the current stream demand.

   @public
   @param amount {number} 
   @return {ReadStream} a reference to this, so the API can be used fluently
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_rabbitMQConsumer["fetch(long)"](__args[0]) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an exception handler on the read stream.

   @public
   @param exceptionHandler {function} the exception handler 
   @return {RabbitMQConsumer} a reference to this, so the API can be used fluently
   */
  this.exceptionHandler =  function(exceptionHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_rabbitMQConsumer["exceptionHandler(io.vertx.core.Handler)"](__args[0] == null ? null : function(jVal) {
        __args[0](utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a message handler. As message appear in a queue, the handler will be called with the message.

   @public
   @param messageArrived {function} 
   @return {RabbitMQConsumer} a reference to this, so the API can be used fluently
   */
  this.handler =  function(messageArrived) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_rabbitMQConsumer["handler(io.vertx.core.Handler)"](__args[0] == null ? null : function(jVal) {
        __args[0](utils.convReturnVertxGen(RabbitMQMessage, jVal));
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pause the stream of incoming messages from queue.
   <p>
   The messages will continue to arrive, but they will be stored in a internal queue.
   If the queue size would exceed the limit provided by , then incoming messages will be discarded.

   @public

   @return {RabbitMQConsumer} a reference to this, so the API can be used fluently
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_rabbitMQConsumer["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Resume reading from a queue. Flushes internal queue.

   @public

   @return {RabbitMQConsumer} a reference to this, so the API can be used fluently
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_rabbitMQConsumer["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an end handler. Once the stream has canceled successfully, the handler will be called.

   @public
   @param endHandler {function} 
   @return {RabbitMQConsumer} a reference to this, so the API can be used fluently
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_rabbitMQConsumer["endHandler(io.vertx.core.Handler)"](__args[0]) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {string} a consumer tag
   */
  this.consumerTag =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_rabbitMQConsumer["consumerTag()"]() ;
    } else if (typeof __super_consumerTag != 'undefined') {
      return __super_consumerTag.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Stop message consumption from a queue.
   <p>
   The operation is asynchronous. When consumption will be stopped, you can by notified via {@link RabbitMQConsumer#endHandler}

   @public
   @param cancelResult {function} contains information about operation status: success/fail. 
   */
  this.cancel =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_rabbitMQConsumer["cancel()"]();
    } else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_rabbitMQConsumer["cancel(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](null, null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_cancel != 'undefined') {
      return __super_cancel.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} is the stream paused?
   */
  this.isPaused =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_rabbitMQConsumer["isPaused()"]() ;
    } else if (typeof __super_isPaused != 'undefined') {
      return __super_isPaused.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_rabbitMQConsumer;
};

RabbitMQConsumer._jclass = utils.getJavaClass("io.vertx.rabbitmq.RabbitMQConsumer");
RabbitMQConsumer._jtype = {accept: function(obj) {
    return RabbitMQConsumer._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(RabbitMQConsumer.prototype, {});
    RabbitMQConsumer.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
RabbitMQConsumer._create = function(jdel) {var obj = Object.create(RabbitMQConsumer.prototype, {});
  RabbitMQConsumer.apply(obj, arguments);
  return obj;
}
module.exports = RabbitMQConsumer;