/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-js/session_store */
var utils = require('vertx-js/util/utils');
var Session = require('vertx-web-js/session');
var Vertx = require('vertx-js/vertx');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JSessionStore = Java.type('io.vertx.ext.web.sstore.SessionStore');

/**
 A session store is used to store sessions for an Vert.x-Web web app

 @class
*/
var SessionStore = function(j_val) {

  var j_sessionStore = j_val;
  var that = this;

  var __super_create = this.create;
  var __super_create = this.create;
  var __super_init = this.init;
  var __super_retryTimeout = this.retryTimeout;
  var __super_createSession = this.createSession;
  var __super_createSession = this.createSession;
  var __super_get = this.get;
  var __super_delete = this.delete;
  var __super_put = this.put;
  var __super_clear = this.clear;
  var __super_size = this.size;
  var __super_close = this.close;
  /**
   Initialize this store.

   @public
   @param vertx {Vertx} the vertx instance 
   @param options {Object} optional Json with extra configuration options 
   @return {SessionStore} self
   */
  this.init =  function(vertx, options) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
      j_sessionStore["init(io.vertx.core.Vertx,io.vertx.core.json.JsonObject)"](__args[0]._jdel, utils.convParamJsonObject(__args[1])) ;
      return that;
    } else if (typeof __super_init != 'undefined') {
      return __super_init.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The retry timeout value in milli seconds used by the session handler when it retrieves a value from the store.<p/>

   A non positive value means there is no retry at all.

   @public

   @return {number} the timeout value, in ms
   */
  this.retryTimeout =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_sessionStore["retryTimeout()"]() ;
    } else if (typeof __super_retryTimeout != 'undefined') {
      return __super_retryTimeout.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Create a new session.

   @public
   @param timeout {number} - the session timeout, in ms 
   @param length {number} - the required length for the session id 
   @return {Session} the session
   */
  this.createSession =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return utils.convReturnVertxGen(Session, j_sessionStore["createSession(long)"](__args[0])) ;
    } else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] ==='number') {
      return utils.convReturnVertxGen(Session, j_sessionStore["createSession(long,int)"](__args[0], __args[1])) ;
    } else if (typeof __super_createSession != 'undefined') {
      return __super_createSession.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the session with the specified ID.

   @public
   @param cookieValue {string} the unique ID of the session 
   @param resultHandler {function} will be called with a result holding the session, or a failure 
   */
  this.get =  function(cookieValue, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_sessionStore["get(java.lang.String,io.vertx.core.Handler)"](__args[0], function(ar) {
        if (ar.succeeded()) {
          __args[1](utils.convReturnVertxGen(Session, ar.result()), null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_get != 'undefined') {
      return __super_get.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Delete the session with the specified ID.

   @public
   @param id {string} the session id 
   @param resultHandler {function} will be called with a success or a failure 
   */
  this.delete =  function(id, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_sessionStore["delete(java.lang.String,io.vertx.core.Handler)"](__args[0], function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_delete != 'undefined') {
      return __super_delete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a session with the specified ID.

   @public
   @param session {Session} the session 
   @param resultHandler {function} will be called with a success or a failure 
   */
  this.put =  function(session, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_sessionStore["put(io.vertx.ext.web.Session,io.vertx.core.Handler)"](__args[0]._jdel, function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_put != 'undefined') {
      return __super_put.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove all sessions from the store.

   @public
   @param resultHandler {function} will be called with a success or a failure 
   */
  this.clear =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_sessionStore["clear(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](null, null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_clear != 'undefined') {
      return __super_clear.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the number of sessions in the store.
   <p>
   Beware of the result which is just an estimate, in particular with distributed session stores.

   @public
   @param resultHandler {function} will be called with the number, or a failure 
   */
  this.size =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_sessionStore["size(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](ar.result(), null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_size != 'undefined') {
      return __super_size.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Close the store

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_sessionStore["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_sessionStore;
};

SessionStore._jclass = utils.getJavaClass("io.vertx.ext.web.sstore.SessionStore");
SessionStore._jtype = {accept: function(obj) {
    return SessionStore._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(SessionStore.prototype, {});
    SessionStore.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
SessionStore._create = function(jdel) {var obj = Object.create(SessionStore.prototype, {});
  SessionStore.apply(obj, arguments);
  return obj;
}
/**
 Create a Session store given a backend and configuration JSON.

 @memberof module:vertx-web-js/session_store
 @param vertx {Vertx} vertx instance 
 @param options {Object} extra options for initialization 
 @return {SessionStore} the store or runtime exception
 */
SessionStore.create =  function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(SessionStore, JSessionStore["create(io.vertx.core.Vertx)"](__args[0]._jdel)) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(SessionStore, JSessionStore["create(io.vertx.core.Vertx,io.vertx.core.json.JsonObject)"](__args[0]._jdel, utils.convParamJsonObject(__args[1]))) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

SessionStore.DEFAULT_SESSIONID_LENGTH = JSessionStore.DEFAULT_SESSIONID_LENGTH;
module.exports = SessionStore;