/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-kafka-client-js/kafka_admin_client */
var utils = require('vertx-js/util/utils');
var Vertx = require('vertx-js/vertx');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JKafkaAdminClient = Java.type('io.vertx.kafka.admin.KafkaAdminClient');
var NewTopic = Java.type('io.vertx.kafka.admin.NewTopic');
var ConsumerGroupListing = Java.type('io.vertx.kafka.admin.ConsumerGroupListing');
var ClusterDescription = Java.type('io.vertx.kafka.admin.ClusterDescription');
var TopicPartition = Java.type('io.vertx.kafka.client.common.TopicPartition');

/**
 Vert.x Kafka Admin client implementation

 @class
*/
var KafkaAdminClient = function(j_val) {

  var j_kafkaAdminClient = j_val;
  var that = this;

  var __super_create = this.create;
  var __super_listTopics = this.listTopics;
  var __super_createTopics = this.createTopics;
  var __super_deleteTopics = this.deleteTopics;
  var __super_listConsumerGroups = this.listConsumerGroups;
  var __super_close = this.close;
  var __super_describeCluster = this.describeCluster;
  var __super_deleteConsumerGroups = this.deleteConsumerGroups;
  var __super_deleteConsumerGroupOffsets = this.deleteConsumerGroupOffsets;
  var __super_close = this.close;
  var __super_close = this.close;
  /**
   List the topics available in the cluster with the default options.

   @public
   @param completionHandler {function} handler called on operation completed with the topics set 
   */
  this.listTopics =  function(completionHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_kafkaAdminClient["listTopics(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](utils.convReturnSet(ar.result()), null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_listTopics != 'undefined') {
      return __super_listTopics.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Creates a batch of new Kafka topics

   @public
   @param topics {Array.<Object>} topics to create 
   @param completionHandler {function} handler called on operation completed 
   */
  this.createTopics =  function(topics, completionHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_kafkaAdminClient["createTopics(java.util.List,io.vertx.core.Handler)"](utils.convParamListDataObject(__args[0], function(json) { return new NewTopic(json); }), function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_createTopics != 'undefined') {
      return __super_createTopics.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Deletes a batch of Kafka topics

   @public
   @param topicNames {Array.<string>} the names of the topics to delete 
   @param completionHandler {function} handler called on operation completed 
   */
  this.deleteTopics =  function(topicNames, completionHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_kafkaAdminClient["deleteTopics(java.util.List,io.vertx.core.Handler)"](utils.convParamListBasicOther(__args[0]), function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_deleteTopics != 'undefined') {
      return __super_deleteTopics.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the the consumer groups available in the cluster with the default options

   @public
   @param completionHandler {function} handler called on operation completed with the consumer groups ids 
   */
  this.listConsumerGroups =  function(completionHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_kafkaAdminClient["listConsumerGroups(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](utils.convReturnListSetDataObject(ar.result()), null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_listConsumerGroups != 'undefined') {
      return __super_listConsumerGroups.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Close the admin client

   @public
   @param timeout {number} timeout to wait for closing 
   @param completionHandler {function} handler called on operation completed 
   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_kafkaAdminClient["close()"]();
    } else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_kafkaAdminClient["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](null, null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'function') {
      j_kafkaAdminClient["close(long,io.vertx.core.Handler)"](__args[0], function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Describe the nodes in the cluster with the default options

   @public
   @param completionHandler {function} handler called on operation completed with the cluster description 
   */
  this.describeCluster =  function(completionHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_kafkaAdminClient["describeCluster(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](utils.convReturnDataObject(ar.result()), null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_describeCluster != 'undefined') {
      return __super_describeCluster.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Delete consumer groups from the cluster.

   @public
   @param groupIds {Array.<string>} the ids of the groups to delete 
   @param completionHandler {function} handler called on operation completed 
   */
  this.deleteConsumerGroups =  function(groupIds, completionHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_kafkaAdminClient["deleteConsumerGroups(java.util.List,io.vertx.core.Handler)"](utils.convParamListBasicOther(__args[0]), function(ar) {
        if (ar.succeeded()) {
          __args[1](null, null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (typeof __super_deleteConsumerGroups != 'undefined') {
      return __super_deleteConsumerGroups.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Delete committed offsets for a set of partitions in a consumer group. This will
   succeed at the partition level only if the group is not actively subscribed
   to the corresponding topic.

   @public
   @param groupId {string} The group id of the group whose offsets will be listed 
   @param partitions {Array.<Object>} The set of partitions in the consumer group whose offsets will be deleted 
   @param completionHandler {function} 
   */
  this.deleteConsumerGroupOffsets =  function(groupId, partitions, completionHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && typeof __args[2] === 'function') {
      j_kafkaAdminClient["deleteConsumerGroupOffsets(java.lang.String,java.util.Set,io.vertx.core.Handler)"](__args[0], utils.convParamSetDataObject(__args[1], function(json) { return new TopicPartition(json); }), function(ar) {
        if (ar.succeeded()) {
          __args[2](null, null);
        } else {
          __args[2](null, ar.cause());
        }
      });
    } else if (typeof __super_deleteConsumerGroupOffsets != 'undefined') {
      return __super_deleteConsumerGroupOffsets.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_kafkaAdminClient;
};

KafkaAdminClient._jclass = utils.getJavaClass("io.vertx.kafka.admin.KafkaAdminClient");
KafkaAdminClient._jtype = {accept: function(obj) {
    return KafkaAdminClient._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(KafkaAdminClient.prototype, {});
    KafkaAdminClient.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
KafkaAdminClient._create = function(jdel) {var obj = Object.create(KafkaAdminClient.prototype, {});
  KafkaAdminClient.apply(obj, arguments);
  return obj;
}
/**
 Create a new KafkaAdminClient instance

 @memberof module:vertx-kafka-client-js/kafka_admin_client
 @param vertx {Vertx} Vert.x instance to use 
 @param config {Object.<string, string>} Kafka admin client configuration 
 @return {KafkaAdminClient} an instance of the KafkaAdminClient
 */
KafkaAdminClient.create =  function(vertx, config) {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(KafkaAdminClient, JKafkaAdminClient["create(io.vertx.core.Vertx,java.util.Map)"](__args[0]._jdel, __args[1])) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = KafkaAdminClient;