/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-mqtt-js/mqtt_endpoint */
var utils = require('vertx-js/util/utils');
var MqttUnsubscribeMessage = require('vertx-mqtt-js/mqtt_unsubscribe_message');
var Buffer = require('vertx-js/buffer');
var MqttSubscribeMessage = require('vertx-mqtt-js/mqtt_subscribe_message');
var MqttPublishMessage = require('vertx-mqtt-js/mqtt_publish_message');
var SocketAddress = require('vertx-js/socket_address');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JMqttEndpoint = Java.type('io.vertx.mqtt.MqttEndpoint');
var MqttAuth = Java.type('io.vertx.mqtt.MqttAuth');
var MqttWill = Java.type('io.vertx.mqtt.MqttWill');

/**
 Represents an MQTT endpoint for point-to-point communication with the remote MQTT client

 @class
*/
var MqttEndpoint = function(j_val) {

  var j_mqttEndpoint = j_val;
  var that = this;

  var __super_close = this.close;
  var __super_remoteAddress = this.remoteAddress;
  var __super_localAddress = this.localAddress;
  var __super_isSsl = this.isSsl;
  var __super_clientIdentifier = this.clientIdentifier;
  var __super_auth = this.auth;
  var __super_will = this.will;
  var __super_protocolVersion = this.protocolVersion;
  var __super_protocolName = this.protocolName;
  var __super_isCleanSession = this.isCleanSession;
  var __super_keepAliveTimeSeconds = this.keepAliveTimeSeconds;
  var __super_lastMessageId = this.lastMessageId;
  var __super_subscriptionAutoAck = this.subscriptionAutoAck;
  var __super_isSubscriptionAutoAck = this.isSubscriptionAutoAck;
  var __super_publishAutoAck = this.publishAutoAck;
  var __super_isPublishAutoAck = this.isPublishAutoAck;
  var __super_autoKeepAlive = this.autoKeepAlive;
  var __super_isAutoKeepAlive = this.isAutoKeepAlive;
  var __super_isConnected = this.isConnected;
  var __super_setClientIdentifier = this.setClientIdentifier;
  var __super_disconnectHandler = this.disconnectHandler;
  var __super_subscribeHandler = this.subscribeHandler;
  var __super_unsubscribeHandler = this.unsubscribeHandler;
  var __super_publishHandler = this.publishHandler;
  var __super_publishAcknowledgeHandler = this.publishAcknowledgeHandler;
  var __super_publishReceivedHandler = this.publishReceivedHandler;
  var __super_publishReleaseHandler = this.publishReleaseHandler;
  var __super_publishCompletionHandler = this.publishCompletionHandler;
  var __super_pingHandler = this.pingHandler;
  var __super_closeHandler = this.closeHandler;
  var __super_exceptionHandler = this.exceptionHandler;
  var __super_accept = this.accept;
  var __super_accept = this.accept;
  var __super_reject = this.reject;
  var __super_subscribeAcknowledge = this.subscribeAcknowledge;
  var __super_unsubscribeAcknowledge = this.unsubscribeAcknowledge;
  var __super_publishAcknowledge = this.publishAcknowledge;
  var __super_publishReceived = this.publishReceived;
  var __super_publishRelease = this.publishRelease;
  var __super_publishComplete = this.publishComplete;
  var __super_publish = this.publish;
  var __super_publish = this.publish;
  var __super_publish = this.publish;
  var __super_pong = this.pong;
  /**
   Close the endpoint, so the connection with remote MQTT client

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttEndpoint["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {SocketAddress} the remote address for this socket
   */
  this.remoteAddress =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedremoteAddress == null) {
        that.cachedremoteAddress = utils.convReturnVertxGen(SocketAddress, j_mqttEndpoint["remoteAddress()"]());
      }
      return that.cachedremoteAddress;
    } else if (typeof __super_remoteAddress != 'undefined') {
      return __super_remoteAddress.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {SocketAddress} the local address for this socket
   */
  this.localAddress =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedlocalAddress == null) {
        that.cachedlocalAddress = utils.convReturnVertxGen(SocketAddress, j_mqttEndpoint["localAddress()"]());
      }
      return that.cachedlocalAddress;
    } else if (typeof __super_localAddress != 'undefined') {
      return __super_localAddress.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} true if this {@link MqttEndpoint} is encrypted via SSL/TLS.
   */
  this.isSsl =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttEndpoint["isSsl()"]() ;
    } else if (typeof __super_isSsl != 'undefined') {
      return __super_isSsl.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {string} the client identifier as provided by the remote MQTT client
   */
  this.clientIdentifier =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedclientIdentifier == null) {
        that.cachedclientIdentifier = j_mqttEndpoint["clientIdentifier()"]();
      }
      return that.cachedclientIdentifier;
    } else if (typeof __super_clientIdentifier != 'undefined') {
      return __super_clientIdentifier.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Object} the Authentication information as provided by the remote MQTT client
   */
  this.auth =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedauth == null) {
        that.cachedauth = utils.convReturnDataObjectAnnotated(j_mqttEndpoint["auth()"]());
      }
      return that.cachedauth;
    } else if (typeof __super_auth != 'undefined') {
      return __super_auth.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Object} the Will information as provided by the remote MQTT client
   */
  this.will =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedwill == null) {
        that.cachedwill = utils.convReturnDataObjectAnnotated(j_mqttEndpoint["will()"]());
      }
      return that.cachedwill;
    } else if (typeof __super_will != 'undefined') {
      return __super_will.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {number} the protocol version required by the remote MQTT client
   */
  this.protocolVersion =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedprotocolVersion == null) {
        that.cachedprotocolVersion = j_mqttEndpoint["protocolVersion()"]();
      }
      return that.cachedprotocolVersion;
    } else if (typeof __super_protocolVersion != 'undefined') {
      return __super_protocolVersion.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {string} the protocol name provided by the remote MQTT client
   */
  this.protocolName =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedprotocolName == null) {
        that.cachedprotocolName = j_mqttEndpoint["protocolName()"]();
      }
      return that.cachedprotocolName;
    } else if (typeof __super_protocolName != 'undefined') {
      return __super_protocolName.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} true when clean session is requested by the remote MQTT client
   */
  this.isCleanSession =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedisCleanSession == null) {
        that.cachedisCleanSession = j_mqttEndpoint["isCleanSession()"]();
      }
      return that.cachedisCleanSession;
    } else if (typeof __super_isCleanSession != 'undefined') {
      return __super_isCleanSession.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {number} the keep alive timeout (in seconds) specified by the remote MQTT client
   */
  this.keepAliveTimeSeconds =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedkeepAliveTimeSeconds == null) {
        that.cachedkeepAliveTimeSeconds = j_mqttEndpoint["keepAliveTimeSeconds()"]();
      }
      return that.cachedkeepAliveTimeSeconds;
    } else if (typeof __super_keepAliveTimeSeconds != 'undefined') {
      return __super_keepAliveTimeSeconds.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {number} the message identifier used for last published message
   */
  this.lastMessageId =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedlastMessageId == null) {
        that.cachedlastMessageId = j_mqttEndpoint["lastMessageId()"]();
      }
      return that.cachedlastMessageId;
    } else if (typeof __super_lastMessageId != 'undefined') {
      return __super_lastMessageId.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Enable/disable subscription/unsubscription requests auto acknowledge

   @public
   @param isSubscriptionAutoAck {boolean} auto acknowledge status 
   */
  this.subscriptionAutoAck =  function(isSubscriptionAutoAck) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_mqttEndpoint["subscriptionAutoAck(boolean)"](isSubscriptionAutoAck);
    } else if (typeof __super_subscriptionAutoAck != 'undefined') {
      return __super_subscriptionAutoAck.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} true when auto acknowledge status for subscription/unsubscription requests
   */
  this.isSubscriptionAutoAck =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttEndpoint["isSubscriptionAutoAck()"]() ;
    } else if (typeof __super_isSubscriptionAutoAck != 'undefined') {
      return __super_isSubscriptionAutoAck.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Enable/disable publishing (in/out) auto acknowledge

   @public
   @param isPublishAutoAck {boolean} auto acknowledge status 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishAutoAck =  function(isPublishAutoAck) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_mqttEndpoint["publishAutoAck(boolean)"](isPublishAutoAck) ;
      return that;
    } else if (typeof __super_publishAutoAck != 'undefined') {
      return __super_publishAutoAck.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} auto acknowledge status for publishing (in/out)
   */
  this.isPublishAutoAck =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttEndpoint["isPublishAutoAck()"]() ;
    } else if (typeof __super_isPublishAutoAck != 'undefined') {
      return __super_isPublishAutoAck.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Enable/disable auto keep alive (sending ping response)

   @public
   @param isAutoKeepAlive {boolean} auto keep alive 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.autoKeepAlive =  function(isAutoKeepAlive) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_mqttEndpoint["autoKeepAlive(boolean)"](isAutoKeepAlive) ;
      return that;
    } else if (typeof __super_autoKeepAlive != 'undefined') {
      return __super_autoKeepAlive.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} the auto keep alive status (sending ping response)
   */
  this.isAutoKeepAlive =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttEndpoint["isAutoKeepAlive()"]() ;
    } else if (typeof __super_isAutoKeepAlive != 'undefined') {
      return __super_isAutoKeepAlive.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} if the connection between remote client and local endpoint is established/open
   */
  this.isConnected =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_mqttEndpoint["isConnected()"]() ;
    } else if (typeof __super_isConnected != 'undefined') {
      return __super_isConnected.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set client identifier if not provided by the remote MQTT client (zero-bytes)

   @public
   @param clientIdentifier {string} the client identifier 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.setClientIdentifier =  function(clientIdentifier) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_mqttEndpoint["setClientIdentifier(java.lang.String)"](clientIdentifier) ;
      return that;
    } else if (typeof __super_setClientIdentifier != 'undefined') {
      return __super_setClientIdentifier.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a disconnect handler on the MQTT endpoint. This handler is called when a DISCONNECT
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.disconnectHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["disconnectHandler(io.vertx.core.Handler)"](handler) ;
      return that;
    } else if (typeof __super_disconnectHandler != 'undefined') {
      return __super_disconnectHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a subscribe handler on the MQTT endpoint. This handler is called when a SUBSCRIBE
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.subscribeHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["subscribeHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(utils.convReturnVertxGen(MqttSubscribeMessage, jVal));
      }) ;
      return that;
    } else if (typeof __super_subscribeHandler != 'undefined') {
      return __super_subscribeHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a unsubscribe handler on the MQTT endpoint. This handler is called when a UNSUBSCRIBE
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.unsubscribeHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["unsubscribeHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(utils.convReturnVertxGen(MqttUnsubscribeMessage, jVal));
      }) ;
      return that;
    } else if (typeof __super_unsubscribeHandler != 'undefined') {
      return __super_unsubscribeHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the publish handler on the MQTT endpoint. This handler is called when a PUBLISH
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["publishHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(utils.convReturnVertxGen(MqttPublishMessage, jVal));
      }) ;
      return that;
    } else if (typeof __super_publishHandler != 'undefined') {
      return __super_publishHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the puback handler on the MQTT endpoint. This handler is called when a PUBACK
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishAcknowledgeHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["publishAcknowledgeHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(jVal);
      }) ;
      return that;
    } else if (typeof __super_publishAcknowledgeHandler != 'undefined') {
      return __super_publishAcknowledgeHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the pubrec handler on the MQTT endpoint. This handler is called when a PUBREC
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishReceivedHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["publishReceivedHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(jVal);
      }) ;
      return that;
    } else if (typeof __super_publishReceivedHandler != 'undefined') {
      return __super_publishReceivedHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the pubrel handler on the MQTT endpoint. This handler is called when a PUBREL
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishReleaseHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["publishReleaseHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(jVal);
      }) ;
      return that;
    } else if (typeof __super_publishReleaseHandler != 'undefined') {
      return __super_publishReleaseHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the pubcomp handler on the MQTT endpoint. This handler is called when a PUBCOMP
   message is received by the remote MQTT client

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishCompletionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["publishCompletionHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(jVal);
      }) ;
      return that;
    } else if (typeof __super_publishCompletionHandler != 'undefined') {
      return __super_publishCompletionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the pingreq handler on the MQTT endpoint. This handler is called when a PINGREQ
   message is received by the remote MQTT client. In any case the endpoint sends the
   PINGRESP internally after executing this handler.

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.pingHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["pingHandler(io.vertx.core.Handler)"](handler) ;
      return that;
    } else if (typeof __super_pingHandler != 'undefined') {
      return __super_pingHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a close handler. This will be called when the MQTT endpoint is closed

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.closeHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["closeHandler(io.vertx.core.Handler)"](handler) ;
      return that;
    } else if (typeof __super_closeHandler != 'undefined') {
      return __super_closeHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an exception handler. This will be called when an error at protocol level happens

   @public
   @param handler {function} the handler 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mqttEndpoint["exceptionHandler(io.vertx.core.Handler)"](function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the CONNACK message to the remote MQTT client with "connection accepted"
   return code. See {@link MqttEndpoint#reject} for refusing connection

   @public
   @param sessionPresent {boolean} if a previous session is present 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.accept =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttEndpoint["accept()"]() ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_mqttEndpoint["accept(boolean)"](__args[0]) ;
      return that;
    } else if (typeof __super_accept != 'undefined') {
      return __super_accept.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the CONNACK message to the remote MQTT client rejecting the connection
   request with specified return code. See {@link MqttEndpoint#accept} for accepting connection

   @public
   @param returnCode {Object} the connect return code 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.reject =  function(returnCode) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_mqttEndpoint["reject(io.netty.handler.codec.mqtt.MqttConnectReturnCode)"](io.netty.handler.codec.mqtt.MqttConnectReturnCode.valueOf(returnCode)) ;
      return that;
    } else if (typeof __super_reject != 'undefined') {
      return __super_reject.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the SUBACK message to the remote MQTT client

   @public
   @param subscribeMessageId {number} identifier of the SUBSCRIBE message to acknowledge 
   @param grantedQoSLevels {Array.<Object>} granted QoS levels for the requested topics 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.subscribeAcknowledge =  function(subscribeMessageId, grantedQoSLevels) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'object' && __args[1] instanceof Array) {
      j_mqttEndpoint["subscribeAcknowledge(int,java.util.List)"](subscribeMessageId, utils.convParamListEnum(grantedQoSLevels, function(val) { return Packages.io.netty.handler.codec.mqtt.MqttQoS.valueOf(val); })) ;
      return that;
    } else if (typeof __super_subscribeAcknowledge != 'undefined') {
      return __super_subscribeAcknowledge.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the UNSUBACK message to the remote MQTT client

   @public
   @param unsubscribeMessageId {number} identifier of the UNSUBSCRIBE message to acknowledge 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.unsubscribeAcknowledge =  function(unsubscribeMessageId) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_mqttEndpoint["unsubscribeAcknowledge(int)"](unsubscribeMessageId) ;
      return that;
    } else if (typeof __super_unsubscribeAcknowledge != 'undefined') {
      return __super_unsubscribeAcknowledge.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBACK message to the remote MQTT client

   @public
   @param publishMessageId {number} identifier of the PUBLISH message to acknowledge 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishAcknowledge =  function(publishMessageId) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_mqttEndpoint["publishAcknowledge(int)"](publishMessageId) ;
      return that;
    } else if (typeof __super_publishAcknowledge != 'undefined') {
      return __super_publishAcknowledge.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBREC message to the remote MQTT client

   @public
   @param publishMessageId {number} identifier of the PUBLISH message to acknowledge 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishReceived =  function(publishMessageId) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_mqttEndpoint["publishReceived(int)"](publishMessageId) ;
      return that;
    } else if (typeof __super_publishReceived != 'undefined') {
      return __super_publishReceived.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBREL message to the remote MQTT client

   @public
   @param publishMessageId {number} identifier of the PUBLISH message to acknowledge 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishRelease =  function(publishMessageId) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_mqttEndpoint["publishRelease(int)"](publishMessageId) ;
      return that;
    } else if (typeof __super_publishRelease != 'undefined') {
      return __super_publishRelease.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBCOMP message to the remote MQTT client

   @public
   @param publishMessageId {number} identifier of the PUBLISH message to acknowledge 
   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.publishComplete =  function(publishMessageId) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_mqttEndpoint["publishComplete(int)"](publishMessageId) ;
      return that;
    } else if (typeof __super_publishComplete != 'undefined') {
      return __super_publishComplete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PUBLISH message to the remote MQTT server explicitly specifying the messageId

   @public
   @param topic {string} topic on which the message is published 
   @param payload {Buffer} message payload 
   @param qosLevel {Object} QoS level 
   @param isDup {boolean} if the message is a duplicate 
   @param isRetain {boolean} if the message needs to be retained 
   @param messageId {number} message ID 
   @param publishSentHandler {function} handler called after PUBLISH packet sent with a packetId 
   @return {MqttEndpoint} current MQTT client instance
   */
  this.publish =  function() {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean') {
      j_mqttEndpoint["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4]) ;
      return that;
    } else if (__args.length === 6 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean' && typeof __args[5] === 'function') {
      j_mqttEndpoint["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean,io.vertx.core.Handler)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4], function(ar) {
        if (ar.succeeded()) {
          __args[5](ar.result(), null);
        } else {
          __args[5](null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mqttEndpoint["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean,io.vertx.core.Handler)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4], function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (__args.length === 7 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean' && typeof __args[5] ==='number' && typeof __args[6] === 'function') {
      j_mqttEndpoint["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean,int,io.vertx.core.Handler)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4], __args[5], function(ar) {
        if (ar.succeeded()) {
          __args[6](ar.result(), null);
        } else {
          __args[6](null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 6 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] === 'string' && typeof __args[3] ==='boolean' && typeof __args[4] ==='boolean' && typeof __args[5] ==='number') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mqttEndpoint["publish(java.lang.String,io.vertx.core.buffer.Buffer,io.netty.handler.codec.mqtt.MqttQoS,boolean,boolean,int,io.vertx.core.Handler)"](__args[0], __args[1]._jdel, io.netty.handler.codec.mqtt.MqttQoS.valueOf(__args[2]), __args[3], __args[4], __args[5], function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_publish != 'undefined') {
      return __super_publish.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends the PINGRESP message to the remote MQTT client

   @public

   @return {MqttEndpoint} a reference to this, so the API can be used fluently
   */
  this.pong =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mqttEndpoint["pong()"]() ;
      return that;
    } else if (typeof __super_pong != 'undefined') {
      return __super_pong.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_mqttEndpoint;
};

MqttEndpoint._jclass = utils.getJavaClass("io.vertx.mqtt.MqttEndpoint");
MqttEndpoint._jtype = {accept: function(obj) {
    return MqttEndpoint._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(MqttEndpoint.prototype, {});
    MqttEndpoint.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
MqttEndpoint._create = function(jdel) {var obj = Object.create(MqttEndpoint.prototype, {});
  MqttEndpoint.apply(obj, arguments);
  return obj;
}
module.exports = MqttEndpoint;