/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-redis-js/redis */
var utils = require('vertx-js/util/utils');
var Pipe = require('vertx-js/pipe');
var Response = require('vertx-redis-js/response');
var WriteStream = require('vertx-js/write_stream');
var Vertx = require('vertx-js/vertx');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');
var Request = require('vertx-redis-js/request');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRedis = Java.type('io.vertx.redis.client.Redis');
var RedisOptions = Java.type('io.vertx.redis.client.RedisOptions');

/**
 A simple Redis client.

 @class
*/
var Redis = function(j_val) {

  var j_redis = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_pipe = this.pipe;
  var __super_pipeTo = this.pipeTo;
  var __super_createClient = this.createClient;
  var __super_createClient = this.createClient;
  var __super_connect = this.connect;
  var __super_exceptionHandler = this.exceptionHandler;
  var __super_handler = this.handler;
  var __super_pause = this.pause;
  var __super_resume = this.resume;
  var __super_fetch = this.fetch;
  var __super_endHandler = this.endHandler;
  var __super_send = this.send;
  var __super_batch = this.batch;
  var __super_socketAddress = this.socketAddress;
  var __super_close = this.close;
  /**
   Pause this stream and return a  to transfer the elements of this stream to a destination .
   <p/>
   The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.

   @public

   @return {Pipe} a pipe
   */
  this.pipe =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Pipe, j_redis["pipe()"](), Response._jtype) ;
    } else if (typeof __super_pipe != 'undefined') {
      return __super_pipe.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   <p>
   Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   <p>
   Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   called with the result.

   @public
   @param dst {WriteStream} the destination write stream 
   @param handler {function} 
   */
  this.pipeTo =  function(dst, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_redis["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_pipeTo != 'undefined') {
      return __super_pipeTo.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Connects to the redis server.

   @public
   @param handler {function} the async result handler 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.connect =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_redis["connect(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(Redis, ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["connect(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Redis, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_connect != 'undefined') {
      return __super_connect.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an exception handler on the read stream.

   @public
   @param handler {function} the exception handler 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redis["exceptionHandler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a data handler. As data is read, the handler will be called with the data.

   @public
   @param handler {function} 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.handler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redis["handler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnVertxGen(Response, jVal));
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pause the <code>ReadStream</code>, it sets the buffer in <code>fetch</code> mode and clears the actual demand.
   <p>
   While it's paused, no data will be sent to the data <code>handler</code>.

   @public

   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redis["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Resume reading, and sets the buffer in <code>flowing</code> mode.
   <p/>
   If the <code>ReadStream</code> has been paused, reading will recommence on it.

   @public

   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redis["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Fetch the specified <code>amount</code> of elements. If the <code>ReadStream</code> has been paused, reading will
   recommence with the specified <code>amount</code> of items, otherwise the specified <code>amount</code> will
   be added to the current stream demand.

   @public
   @param amount {number} 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_redis["fetch(long)"](amount) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an end handler. Once the stream has ended, and there is no more data to be read, this handler will be called.

   @public
   @param endHandler {function} 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redis["endHandler(io.vertx.core.Handler)"](endHandler) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Send the given command to the redis server or cluster.

   @public
   @param command {Request} the command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {Redis} fluent self.
   */
  this.send =  function(command, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_redis["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_send != 'undefined') {
      return __super_send.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends a list of commands in a single IO operation, this prevents any inter twinning to happen from other
   client users.

   @public
   @param commands {Array.<Request>} list of command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {Redis} fluent self.
   */
  this.batch =  function(commands, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_redis["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_batch != 'undefined') {
      return __super_batch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns the address associated with this client.

   @public

   @return {string} the address.
   */
  this.socketAddress =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_redis["socketAddress()"]() ;
    } else if (typeof __super_socketAddress != 'undefined') {
      return __super_socketAddress.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redis["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_redis;
};

Redis._jclass = utils.getJavaClass("io.vertx.redis.client.Redis");
Redis._jtype = {accept: function(obj) {
    return Redis._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Redis.prototype, {});
    Redis.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Redis._create = function(jdel) {var obj = Object.create(Redis.prototype, {});
  Redis.apply(obj, arguments);
  return obj;
}
/**
 Connect to redis, the <code>onConnect</code> will get the {@link Redis} instance.

 @memberof module:vertx-redis-js/redis
 @param vertx {Vertx} 
 @param options {Object} 
 @return {Redis}
 */
Redis.createClient =  function() {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
    return utils.convReturnVertxGen(Redis, JRedis["createClient(io.vertx.core.Vertx,java.lang.String)"](__args[0]._jdel, __args[1])) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(Redis, JRedis["createClient(io.vertx.core.Vertx,io.vertx.redis.client.RedisOptions)"](__args[0]._jdel, __args[1]  != null ? new RedisOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = Redis;