/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-cassandra-js/cassandra_client */
var utils = require('vertx-js/util/utils');
var Vertx = require('vertx-js/vertx');
var ResultSet = require('vertx-cassandra-js/result_set');
var CassandraRowStream = require('vertx-cassandra-js/cassandra_row_stream');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JCassandraClient = Java.type('io.vertx.cassandra.CassandraClient');
var CassandraClientOptions = Java.type('io.vertx.cassandra.CassandraClientOptions');

/**
 Eclipse Vert.x Cassandra client.

 @class
*/
var CassandraClient = function(j_val) {

  var j_cassandraClient = j_val;
  var that = this;

  var __super_createNonShared = this.createNonShared;
  var __super_createNonShared = this.createNonShared;
  var __super_createShared = this.createShared;
  var __super_createShared = this.createShared;
  var __super_createShared = this.createShared;
  var __super_createShared = this.createShared;
  var __super_isConnected = this.isConnected;
  var __super_execute = this.execute;
  var __super_queryStream = this.queryStream;
  var __super_close = this.close;
  var __super_close = this.close;
  /**

   @public

   @return {boolean} whether this Cassandra client instance is connected
   */
  this.isConnected =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_cassandraClient["isConnected()"]() ;
    } else if (typeof __super_isConnected != 'undefined') {
      return __super_isConnected.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Execute the query and provide a handler for consuming results.

   @public
   @param query {string} the query to execute 
   @param resultHandler {function} handler called when result of execution is present, but can be not fully fetched 
   @return {CassandraClient} current Cassandra client instance
   */
  this.execute =  function(query, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_cassandraClient["execute(java.lang.String,io.vertx.core.Handler)"](query, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnVertxGen(ResultSet, ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_cassandraClient["execute(java.lang.String,io.vertx.core.Handler)"](query, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(ResultSet, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_execute != 'undefined') {
      return __super_execute.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Executes the given SQL <code>SELECT</code> statement which returns the results of the query as a read stream.

   @public
   @param sql {string} the SQL to execute. For example <code>SELECT * FROM table ...</code>. 
   @param rowStreamHandler {function} the handler which is called once the operation completes. It will return an instance of {@link CassandraRowStream}. 
   @return {CassandraClient} current Cassandra client instance
   */
  this.queryStream =  function(sql, rowStreamHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_cassandraClient["queryStream(java.lang.String,io.vertx.core.Handler)"](sql, function(ar) {
        if (ar.succeeded()) {
          rowStreamHandler(utils.convReturnVertxGen(CassandraRowStream, ar.result()), null);
        } else {
          rowStreamHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_cassandraClient["queryStream(java.lang.String,io.vertx.core.Handler)"](sql, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(CassandraRowStream, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_queryStream != 'undefined') {
      return __super_queryStream.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Closes this client.

   @public
   @param closeHandler {function} handler called when client is closed 
   @return {CassandraClient} current Cassandra client instance
   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_cassandraClient["close()"]() ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_cassandraClient["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](null, null);
        } else {
          __args[0](null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_cassandraClient["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_cassandraClient;
};

CassandraClient._jclass = utils.getJavaClass("io.vertx.cassandra.CassandraClient");
CassandraClient._jtype = {accept: function(obj) {
    return CassandraClient._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(CassandraClient.prototype, {});
    CassandraClient.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
CassandraClient._create = function(jdel) {var obj = Object.create(CassandraClient.prototype, {});
  CassandraClient.apply(obj, arguments);
  return obj;
}
/**
 Create a Cassandra client which maintains its own driver session.
 <p>
 It is not recommended to create several non shared clients in an application.

 @memberof module:vertx-cassandra-js/cassandra_client
 @param vertx {Vertx} the Vert.x instance 
 @param options {Object} the options 
 @return {CassandraClient} the client
 */
CassandraClient.createNonShared =  function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createNonShared(io.vertx.core.Vertx)"](__args[0]._jdel)) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createNonShared(io.vertx.core.Vertx,io.vertx.cassandra.CassandraClientOptions)"](__args[0]._jdel, __args[1]  != null ? new CassandraClientOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a Cassandra client that shares its driver session with any other client having the same name.

 @memberof module:vertx-cassandra-js/cassandra_client
 @param vertx {Vertx} the Vert.x instance 
 @param clientName {string} the shared client name 
 @param options {Object} the options 
 @return {CassandraClient} the client
 */
CassandraClient.createShared =  function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createShared(io.vertx.core.Vertx)"](__args[0]._jdel)) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createShared(io.vertx.core.Vertx,java.lang.String)"](__args[0]._jdel, __args[1])) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createShared(io.vertx.core.Vertx,io.vertx.cassandra.CassandraClientOptions)"](__args[0]._jdel, __args[1]  != null ? new CassandraClientOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null)) ;
  } else if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && (typeof __args[2] === 'object' && __args[2] != null)) {
    return utils.convReturnVertxGen(CassandraClient, JCassandraClient["createShared(io.vertx.core.Vertx,java.lang.String,io.vertx.cassandra.CassandraClientOptions)"](__args[0]._jdel, __args[1], __args[2]  != null ? new CassandraClientOptions(new JsonObject(Java.asJSONCompatible(__args[2]))) : null)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

CassandraClient.DEFAULT_SHARED_CLIENT_NAME = JCassandraClient.DEFAULT_SHARED_CLIENT_NAME;
module.exports = CassandraClient;