/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/json_pointer_iterator */
var utils = require('vertx-js/util/utils');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JJsonPointerIterator = Java.type('io.vertx.core.json.pointer.JsonPointerIterator');

/**

 @class
*/
var JsonPointerIterator = function(j_val) {

  var j_jsonPointerIterator = j_val;
  var that = this;

  var __super_isObject = this.isObject;
  var __super_isArray = this.isArray;
  var __super_isNull = this.isNull;
  var __super_objectContainsKey = this.objectContainsKey;
  var __super_getObjectParameter = this.getObjectParameter;
  var __super_getArrayElement = this.getArrayElement;
  var __super_writeObjectParameter = this.writeObjectParameter;
  var __super_writeArrayElement = this.writeArrayElement;
  var __super_appendArrayElement = this.appendArrayElement;
  /**

   @public
   @param currentValue {Object} 
   @return {boolean} <code>true</code> if the current value is a queryable object
   */
  this.isObject =  function(currentValue) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      return j_jsonPointerIterator["isObject(java.lang.Object)"](utils.convParamTypeUnknown(currentValue)) ;
    } else if (typeof __super_isObject != 'undefined') {
      return __super_isObject.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param currentValue {Object} 
   @return {boolean} <code>true</code> if the current value is a queryable array
   */
  this.isArray =  function(currentValue) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      return j_jsonPointerIterator["isArray(java.lang.Object)"](utils.convParamTypeUnknown(currentValue)) ;
    } else if (typeof __super_isArray != 'undefined') {
      return __super_isArray.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param currentValue {Object} 
   @return {boolean} <code>true</code> if the current value is null/empty
   */
  this.isNull =  function(currentValue) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      return j_jsonPointerIterator["isNull(java.lang.Object)"](utils.convParamTypeUnknown(currentValue)) ;
    } else if (typeof __super_isNull != 'undefined') {
      return __super_isNull.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param currentValue {Object} 
   @param key {string} object key 
   @return {boolean} <code>true</code> if current value is a queryable object that contains the specified key
   */
  this.objectContainsKey =  function(currentValue, key) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] === 'string') {
      return j_jsonPointerIterator["objectContainsKey(java.lang.Object,java.lang.String)"](utils.convParamTypeUnknown(currentValue), key) ;
    } else if (typeof __super_objectContainsKey != 'undefined') {
      return __super_objectContainsKey.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns the object parameter with specified key.

   @public
   @param currentValue {Object} 
   @param key {string} object key 
   @param createOnMissing {boolean} If the current value is an object that doesn't contain the key, put an empty object at provided key 
   @return {Object} the requested object parameter, or null if the method was not able to find it
   */
  this.getObjectParameter =  function(currentValue, key, createOnMissing) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] !== 'function' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      return utils.convReturnTypeUnknown(j_jsonPointerIterator["getObjectParameter(java.lang.Object,java.lang.String,boolean)"](utils.convParamTypeUnknown(currentValue), key, createOnMissing)) ;
    } else if (typeof __super_getObjectParameter != 'undefined') {
      return __super_getObjectParameter.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Move the iterator the the array element at specified index

   @public
   @param currentValue {Object} 
   @param i {number} array index 
   @return {Object} the request array element, or null if the method was not able to find it
   */
  this.getArrayElement =  function(currentValue, i) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] ==='number') {
      return utils.convReturnTypeUnknown(j_jsonPointerIterator["getArrayElement(java.lang.Object,int)"](utils.convParamTypeUnknown(currentValue), i)) ;
    } else if (typeof __super_getArrayElement != 'undefined') {
      return __super_getArrayElement.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Write object parameter at specified key

   @public
   @param currentValue {Object} 
   @param key {string} 
   @param value {Object} 
   @return {boolean} true if the operation is successful
   */
  this.writeObjectParameter =  function(currentValue, key, value) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] !== 'function' && typeof __args[1] === 'string' && typeof __args[2] !== 'function') {
      return j_jsonPointerIterator["writeObjectParameter(java.lang.Object,java.lang.String,java.lang.Object)"](utils.convParamTypeUnknown(currentValue), key, utils.convParamTypeUnknown(value)) ;
    } else if (typeof __super_writeObjectParameter != 'undefined') {
      return __super_writeObjectParameter.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Write array element at specified index

   @public
   @param currentValue {Object} 
   @param i {number} 
   @param value {Object} 
   @return {boolean} true if the operation is successful
   */
  this.writeArrayElement =  function(currentValue, i, value) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] !== 'function' && typeof __args[1] ==='number' && typeof __args[2] !== 'function') {
      return j_jsonPointerIterator["writeArrayElement(java.lang.Object,int,java.lang.Object)"](utils.convParamTypeUnknown(currentValue), i, utils.convParamTypeUnknown(value)) ;
    } else if (typeof __super_writeArrayElement != 'undefined') {
      return __super_writeArrayElement.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Append array element

   @public
   @param currentValue {Object} 
   @param value {Object} 
   @return {boolean} true if the operation is successful
   */
  this.appendArrayElement =  function(currentValue, value) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] !== 'function') {
      return j_jsonPointerIterator["appendArrayElement(java.lang.Object,java.lang.Object)"](utils.convParamTypeUnknown(currentValue), utils.convParamTypeUnknown(value)) ;
    } else if (typeof __super_appendArrayElement != 'undefined') {
      return __super_appendArrayElement.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_jsonPointerIterator;
};

JsonPointerIterator._jclass = utils.getJavaClass("io.vertx.core.json.pointer.JsonPointerIterator");
JsonPointerIterator._jtype = {accept: function(obj) {
    return JsonPointerIterator._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(JsonPointerIterator.prototype, {});
    JsonPointerIterator.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
JsonPointerIterator._create = function(jdel) {var obj = Object.create(JsonPointerIterator.prototype, {});
  JsonPointerIterator.apply(obj, arguments);
  return obj;
}
JsonPointerIterator.JSON_ITERATOR = utils.convReturnVertxGen(JsonPointerIterator, JJsonPointerIterator.JSON_ITERATOR);
module.exports = JsonPointerIterator;