/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-client-js/http_request */
var utils = require('vertx-js/util/utils');
var ResponsePredicate = require('vertx-web-client-js/response_predicate');
var MultipartForm = require('vertx-web-common-js/multipart_form');
var HttpResponse = require('vertx-web-client-js/http_response');
var Buffer = require('vertx-js/buffer');
var BodyCodec = require('vertx-web-common-js/body_codec');
var MultiMap = require('vertx-js/multi_map');
var ReadStream = require('vertx-js/read_stream');
var ResponsePredicateResult = require('vertx-web-client-js/response_predicate_result');
var Future = require('vertx-js/future');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JHttpRequest = Java.type('io.vertx.ext.web.client.HttpRequest');

/**
 A client-side HTTP request.
 <p>
 @class
*/
var HttpRequest = function(j_val, j_arg_0) {

  var j_httpRequest = j_val;
  var that = this;
  var j_T = typeof j_arg_0 !== 'undefined' ? j_arg_0 : utils.unknown_jtype;
  var __super_method = this.method;
  var __super_rawMethod = this.rawMethod;
  var __super_port = this.port;
  var __super_as = this.as;
  var __super_host = this.host;
  var __super_virtualHost = this.virtualHost;
  var __super_uri = this.uri;
  var __super_putHeaders = this.putHeaders;
  var __super_putHeader = this.putHeader;
  var __super_headers = this.headers;
  var __super_basicAuthentication = this.basicAuthentication;
  var __super_basicAuthentication = this.basicAuthentication;
  var __super_bearerTokenAuthentication = this.bearerTokenAuthentication;
  var __super_ssl = this.ssl;
  var __super_timeout = this.timeout;
  var __super_addQueryParam = this.addQueryParam;
  var __super_setQueryParam = this.setQueryParam;
  var __super_followRedirects = this.followRedirects;
  var __super_expect = this.expect;
  var __super_expect = this.expect;
  var __super_queryParams = this.queryParams;
  var __super_copy = this.copy;
  var __super_multipartMixed = this.multipartMixed;
  var __super_sendStream = this.sendStream;
  var __super_sendBuffer = this.sendBuffer;
  var __super_sendJsonObject = this.sendJsonObject;
  var __super_sendJson = this.sendJson;
  var __super_sendForm = this.sendForm;
  var __super_sendMultipartForm = this.sendMultipartForm;
  var __super_send = this.send;
  /**
   Configure the request to use a new method <code>value</code>.

   @public
   @param value {Object} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.method =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["method(io.vertx.core.http.HttpMethod)"](io.vertx.core.http.HttpMethod.valueOf(value)) ;
      return that;
    } else if (typeof __super_method != 'undefined') {
      return __super_method.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to use a custom HTTP method

   @public
   @param method {string} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.rawMethod =  function(method) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["rawMethod(java.lang.String)"](method) ;
      return that;
    } else if (typeof __super_rawMethod != 'undefined') {
      return __super_rawMethod.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to use a new port <code>value</code>.

   @public
   @param value {number} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.port =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_httpRequest["port(int)"](value) ;
      return that;
    } else if (typeof __super_port != 'undefined') {
      return __super_port.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to decode the response with the <code>responseCodec</code>.

   @public
   @param responseCodec {BodyCodec} the response codec 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.as =  function(responseCodec) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      return utils.convReturnVertxGen(HttpRequest, j_httpRequest["as(io.vertx.ext.web.codec.BodyCodec)"](responseCodec._jdel), undefined) ;
    } else if (typeof __super_as != 'undefined') {
      return __super_as.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to use a new host <code>value</code>.

   @public
   @param value {string} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.host =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["host(java.lang.String)"](value) ;
      return that;
    } else if (typeof __super_host != 'undefined') {
      return __super_host.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to use a virtual host <code>value</code>.
   <p/>
   Usually the header <i>host</i> (<i>:authority</i> pseudo header for HTTP/2) is set from the request host value
   since this host value resolves to the server IP address.
   <p/>
   Sometimes you need to set a host header for an address that does not resolve to the server IP address.
   The virtual host value overrides the value of the actual <i>host</i> header (<i>:authority</i> pseudo header
   for HTTP/2).
   <p/>
   The virtual host is also be used for SNI.

   @public
   @param value {string} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.virtualHost =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["virtualHost(java.lang.String)"](value) ;
      return that;
    } else if (typeof __super_virtualHost != 'undefined') {
      return __super_virtualHost.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to use a new request URI <code>value</code>.
   <p>
   When the uri has query parameters, they are set in the {@link HttpRequest#queryParams} multimap, overwritting
   any parameters previously set.

   @public
   @param value {string} 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.uri =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["uri(java.lang.String)"](value) ;
      return that;
    } else if (typeof __super_uri != 'undefined') {
      return __super_uri.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to add multiple HTTP headers .

   @public
   @param headers {MultiMap} The HTTP headers 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.putHeaders =  function(headers) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_httpRequest["putHeaders(io.vertx.core.MultiMap)"](headers._jdel) ;
      return that;
    } else if (typeof __super_putHeaders != 'undefined') {
      return __super_putHeaders.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to add a new HTTP header.

   @public
   @param name {string} the header name 
   @param value {string} the header value 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.putHeader =  function(name, value) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpRequest["putHeader(java.lang.String,java.lang.String)"](name, value) ;
      return that;
    } else if (typeof __super_putHeader != 'undefined') {
      return __super_putHeader.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {MultiMap} The HTTP headers
   */
  this.headers =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedheaders == null) {
        that.cachedheaders = utils.convReturnVertxGen(MultiMap, j_httpRequest["headers()"]());
      }
      return that.cachedheaders;
    } else if (typeof __super_headers != 'undefined') {
      return __super_headers.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to perform basic access authentication.
   <p>
   In basic HTTP authentication, a request contains a header field of the form 'Authorization: Basic &#60;credentials&#62;',
   where credentials is the base64 encoding of id and password joined by a colon.
   </p>

   @public
   @param id {Buffer} the id 
   @param password {Buffer} the password 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.basicAuthentication =  function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpRequest["basicAuthentication(java.lang.String,java.lang.String)"](__args[0], __args[1]) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'object' && __args[1]._jdel) {
      j_httpRequest["basicAuthentication(io.vertx.core.buffer.Buffer,io.vertx.core.buffer.Buffer)"](__args[0]._jdel, __args[1]._jdel) ;
      return that;
    } else if (typeof __super_basicAuthentication != 'undefined') {
      return __super_basicAuthentication.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configure the request to perform bearer token authentication.
   <p>
   In OAuth 2.0, a request contains a header field of the form 'Authorization: Bearer &#60;bearerToken&#62;',
   where bearerToken is the bearer token issued by an authorization server to access protected resources.
   </p>

   @public
   @param bearerToken {string} the bearer token 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.bearerTokenAuthentication =  function(bearerToken) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_httpRequest["bearerTokenAuthentication(java.lang.String)"](bearerToken) ;
      return that;
    } else if (typeof __super_bearerTokenAuthentication != 'undefined') {
      return __super_bearerTokenAuthentication.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param value {boolean} 
   @return {HttpRequest}
   */
  this.ssl =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_httpRequest["ssl(java.lang.Boolean)"](value) ;
      return that;
    } else if (typeof __super_ssl != 'undefined') {
      return __super_ssl.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Configures the amount of time in milliseconds after which if the request does not return any data within the timeout
   period an TimeoutException fails the request.
   <p>
   Setting zero or a negative <code>value</code> disables the timeout.

   @public
   @param value {number} The quantity of time in milliseconds. 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.timeout =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_httpRequest["timeout(long)"](value) ;
      return that;
    } else if (typeof __super_timeout != 'undefined') {
      return __super_timeout.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameter to the request.

   @public
   @param paramName {string} the param name 
   @param paramValue {string} the param value 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.addQueryParam =  function(paramName, paramValue) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpRequest["addQueryParam(java.lang.String,java.lang.String)"](paramName, paramValue) ;
      return that;
    } else if (typeof __super_addQueryParam != 'undefined') {
      return __super_addQueryParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set a query parameter to the request.

   @public
   @param paramName {string} the param name 
   @param paramValue {string} the param value 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.setQueryParam =  function(paramName, paramValue) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_httpRequest["setQueryParam(java.lang.String,java.lang.String)"](paramName, paramValue) ;
      return that;
    } else if (typeof __super_setQueryParam != 'undefined') {
      return __super_setQueryParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set wether or not to follow the directs for the request.

   @public
   @param value {boolean} true if redirections should be followed 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.followRedirects =  function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_httpRequest["followRedirects(boolean)"](value) ;
      return that;
    } else if (typeof __super_followRedirects != 'undefined') {
      return __super_followRedirects.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add an expectation that the response is valid according to the provided <code>predicate</code>.
   <p>
   Multiple predicates can be added.

   @public
   @param predicate {ResponsePredicate} the predicate 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.expect =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpRequest["expect(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](utils.convReturnVertxGen(HttpResponse, jVal, undefined));
        return jRet._jdel;
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_httpRequest["expect(io.vertx.ext.web.client.predicate.ResponsePredicate)"](__args[0]._jdel) ;
      return that;
    } else if (typeof __super_expect != 'undefined') {
      return __super_expect.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return the current query parameters.

   @public

   @return {MultiMap} the current query parameters
   */
  this.queryParams =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(MultiMap, j_httpRequest["queryParams()"]()) ;
    } else if (typeof __super_queryParams != 'undefined') {
      return __super_queryParams.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Copy this request

   @public

   @return {HttpRequest} a copy of this request
   */
  this.copy =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(HttpRequest, j_httpRequest["copy()"](), undefined) ;
    } else if (typeof __super_copy != 'undefined') {
      return __super_copy.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Allow or disallow multipart mixed encoding when sending  having files sharing the same
   file name.
   <br/>
   The default value is <code>true</code>.
   <br/>
   Set to <code>false</code> if you want to achieve the behavior for <a href="http://www.w3.org/TR/html5/forms.html#multipart-form-data">HTML5</a>.

   @public
   @param allow {boolean} <code>true</code> allows use of multipart mixed encoding 
   @return {HttpRequest} a reference to this, so the API can be used fluently
   */
  this.multipartMixed =  function(allow) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      return utils.convReturnVertxGen(HttpRequest, j_httpRequest["multipartMixed(boolean)"](allow), undefined) ;
    } else if (typeof __super_multipartMixed != 'undefined') {
      return __super_multipartMixed.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> stream.

   @public
   @param body {ReadStream} the body 
   @param handler {function} 
   */
  this.sendStream =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_httpRequest["sendStream(io.vertx.core.streams.ReadStream,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendStream(io.vertx.core.streams.ReadStream,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendStream != 'undefined') {
      return __super_sendStream.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> buffer.

   @public
   @param body {Buffer} the body 
   @param handler {function} 
   */
  this.sendBuffer =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_httpRequest["sendBuffer(io.vertx.core.buffer.Buffer,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendBuffer(io.vertx.core.buffer.Buffer,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendBuffer != 'undefined') {
      return __super_sendBuffer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   set to <code>application/json</code>.

   @public
   @param body {Object} the body 
   @param handler {function} 
   */
  this.sendJsonObject =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && (typeof __args[0] === 'object' && __args[0] != null) && typeof __args[1] === 'function') {
      j_httpRequest["sendJsonObject(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(body), function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendJsonObject(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(body), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendJsonObject != 'undefined') {
      return __super_sendJsonObject.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   set to <code>application/json</code>.

   @public
   @param body {Object} the body 
   @param handler {function} 
   */
  this.sendJson =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] === 'function') {
      j_httpRequest["sendJson(java.lang.Object,io.vertx.core.Handler)"](utils.convParamTypeUnknown(body), function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] !== 'function') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendJson(java.lang.Object,io.vertx.core.Handler)"](utils.convParamTypeUnknown(body), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendJson != 'undefined') {
      return __super_sendJson.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> multimap encoded as form and the content type
   set to <code>application/x-www-form-urlencoded</code>.
   <p>
   When the content type header is previously set to <code>multipart/form-data</code> it will be used instead.

   @public
   @param body {MultiMap} the body 
   @param handler {function} 
   */
  this.sendForm =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_httpRequest["sendForm(io.vertx.core.MultiMap,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendForm(io.vertx.core.MultiMap,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendForm != 'undefined') {
      return __super_sendForm.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link HttpRequest#send} but with an HTTP request <code>body</code> multimap encoded as form and the content type
   set to <code>multipart/form-data</code>. You may use this method to send attributes and upload files.

   @public
   @param body {MultipartForm} the body 
   @param handler {function} 
   */
  this.sendMultipartForm =  function(body, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_httpRequest["sendMultipartForm(io.vertx.ext.web.multipart.MultipartForm,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["sendMultipartForm(io.vertx.ext.web.multipart.MultipartForm,io.vertx.core.Handler)"](body._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_sendMultipartForm != 'undefined') {
      return __super_sendMultipartForm.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Send a request, the <code>handler</code> will receive the response as an {@link HttpResponse}.

   @public
   @param handler {function} 
   */
  this.send =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_httpRequest["send(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_httpRequest["send(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(HttpResponse, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_send != 'undefined') {
      return __super_send.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_httpRequest;
};

HttpRequest._jclass = utils.getJavaClass("io.vertx.ext.web.client.HttpRequest");
HttpRequest._jtype = {accept: function(obj) {
    return HttpRequest._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(HttpRequest.prototype, {});
    HttpRequest.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
HttpRequest._create = function(jdel) {var obj = Object.create(HttpRequest.prototype, {});
  HttpRequest.apply(obj, arguments);
  return obj;
}
module.exports = HttpRequest;