/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-auth-common-js/hashing_strategy */
var utils = require('vertx-js/util/utils');
var HashingAlgorithm = require('vertx-auth-common-js/hashing_algorithm');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JHashingStrategy = Java.type('io.vertx.ext.auth.HashingStrategy');

/**
 Hashing Strategy manager.

 This class will load system provided hashing strategies and algorithms.

 @class
*/
var HashingStrategy = function(j_val) {

  var j_hashingStrategy = j_val;
  var that = this;

  var __super_load = this.load;
  var __super_hash = this.hash;
  var __super_verify = this.verify;
  var __super_get = this.get;
  var __super_put = this.put;
  /**
   Hashes a password.

   @public
   @param id {string} the algorithm id 
   @param params {Object.<string, string>} the algorithm specific paramters 
   @param salt {string} the given salt 
   @param password {string} the given password 
   @return {string} the hashed string
   */
  this.hash =  function(id, params, salt, password) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'string' && typeof __args[3] === 'string') {
      return j_hashingStrategy["hash(java.lang.String,java.util.Map,java.lang.String,java.lang.String)"](id, params, salt, password) ;
    } else if (typeof __super_hash != 'undefined') {
      return __super_hash.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Time constant password check. Regardless of the check, this algorithm executes the same number of
   checks regardless of the correctly number of characters

   @public
   @param hash {string} the hash to verify 
   @param password {string} the password to test against 
   @return {boolean} boolean
   */
  this.verify =  function(hash, password) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      return j_hashingStrategy["verify(java.lang.String,java.lang.String)"](hash, password) ;
    } else if (typeof __super_verify != 'undefined') {
      return __super_verify.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get an algorithm interface by its Id

   @public
   @param id {string} the algorithm id 
   @return {HashingAlgorithm} the algorithm
   */
  this.get =  function(id) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(HashingAlgorithm, j_hashingStrategy["get(java.lang.String)"](id)) ;
    } else if (typeof __super_get != 'undefined') {
      return __super_get.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Put or replace an algorithm into the list of system loaded algorithms.

   @public
   @param id {string} the algorithm id 
   @param algorithm {HashingAlgorithm} the implementation 
   @return {HashingStrategy} self
   */
  this.put =  function(id, algorithm) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel) {
      j_hashingStrategy["put(java.lang.String,io.vertx.ext.auth.HashingAlgorithm)"](id, algorithm._jdel) ;
      return that;
    } else if (typeof __super_put != 'undefined') {
      return __super_put.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_hashingStrategy;
};

HashingStrategy._jclass = utils.getJavaClass("io.vertx.ext.auth.HashingStrategy");
HashingStrategy._jtype = {accept: function(obj) {
    return HashingStrategy._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(HashingStrategy.prototype, {});
    HashingStrategy.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
HashingStrategy._create = function(jdel) {var obj = Object.create(HashingStrategy.prototype, {});
  HashingStrategy.apply(obj, arguments);
  return obj;
}
/**
 Factory method to load the algorithms from the system

 @memberof module:vertx-auth-common-js/hashing_strategy

 @return {HashingStrategy} a Hashing Strategy capable of hashing using the available algorithms
 */
HashingStrategy.load =  function() {
  var __args = arguments;
  if (__args.length === 0) {
    return utils.convReturnVertxGen(HashingStrategy, JHashingStrategy["load()"]()) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = HashingStrategy;