/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-sql-client-js/cursor */
var utils = require('vertx-js/util/utils');
var RowSet = require('vertx-sql-client-js/row_set');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JCursor = Java.type('io.vertx.sqlclient.Cursor');

/**
 A cursor that reads progressively rows from the database, it is useful for reading very large result sets.

 @class
*/
var Cursor = function(j_val) {

  var j_cursor = j_val;
  var that = this;

  var __super_read = this.read;
  var __super_hasMore = this.hasMore;
  var __super_close = this.close;
  var __super_close = this.close;
  /**
   Read rows from the cursor, the result is provided asynchronously to the <code>handler</code>.

   @public
   @param count {number} the amount of rows to read 
   @param handler {function} the handler for the result 
   */
  this.read =  function(count, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'function') {
      j_cursor["read(int,io.vertx.core.Handler)"](count, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(RowSet, ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] ==='number') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_cursor["read(int,io.vertx.core.Handler)"](count, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(RowSet, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_read != 'undefined') {
      return __super_read.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns <code>true</code> when the cursor has results in progress and the {@link Cursor#read} should be called to retrieve
   them.

   @public

   @return {boolean} whether the cursor has more results,
   */
  this.hasMore =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_cursor["hasMore()"]() ;
    } else if (typeof __super_hasMore != 'undefined') {
      return __super_hasMore.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like {@link Cursor#close} but with a <code>completionHandler</code> called when the cursor has been released.

   @public
   @param completionHandler {function} 
   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_cursor["close()"]();
    } else if (__args.length === 1 && typeof __args[0] === 'function') {
      j_cursor["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](null, null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_cursor["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_cursor;
};

Cursor._jclass = utils.getJavaClass("io.vertx.sqlclient.Cursor");
Cursor._jtype = {accept: function(obj) {
    return Cursor._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Cursor.prototype, {});
    Cursor.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Cursor._create = function(jdel) {var obj = Object.create(Cursor.prototype, {});
  Cursor.apply(obj, arguments);
  return obj;
}
module.exports = Cursor;