/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-api-contract-js/http_request_validation_handler */
var utils = require('vertx-js/util/utils');
var CustomValidator = require('vertx-web-api-contract-js/custom_validator');
var ParameterTypeValidator = require('vertx-web-api-contract-js/parameter_type_validator');
var RoutingContext = require('vertx-web-js/routing_context');
var ValidationHandler = require('vertx-web-api-contract-js/validation_handler');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JHTTPRequestValidationHandler = Java.type('io.vertx.ext.web.api.validation.HTTPRequestValidationHandler');

/**
 An interface for add HTTP Request validation. This class can validate parameters inside query, path, headers an
 body (watch below)
 <br/>
 You can assign multiple body type at the same time(for example a JSON schema together with a XML schema). This
 interface support:
 <ul>
 <li>application/x-www-form-urlencoded</li>
 <li>multipart/form-data</li>
 <li>application/xml</li>
 <li>application/json</li>
 </ul>
 Also you can add a form parameter for validation without care about content type of your request. For form
 parameters this interface support both "multipart/form-data" and "application/x-www-form-urlencoded"
 <br/>
 This interface allow extra parameters in the request, so it doesn't care if in a request there's a parameter
 without a specified validation rule
 <br/>
 If a parameter is flagged as an array, it will be validated also if the size of array is 1 element

 @class
*/
var HTTPRequestValidationHandler = function(j_val) {

  var j_hTTPRequestValidationHandler = j_val;
  var that = this;
  ValidationHandler.call(this, j_val);

  var __super_handle = this.handle;
  var __super_create = this.create;
  var __super_addPathParam = this.addPathParam;
  var __super_addPathParamWithPattern = this.addPathParamWithPattern;
  var __super_addPathParamWithCustomTypeValidator = this.addPathParamWithCustomTypeValidator;
  var __super_addQueryParam = this.addQueryParam;
  var __super_addQueryParamWithPattern = this.addQueryParamWithPattern;
  var __super_addQueryParamsArray = this.addQueryParamsArray;
  var __super_addQueryParamsArrayWithPattern = this.addQueryParamsArrayWithPattern;
  var __super_addQueryParamWithCustomTypeValidator = this.addQueryParamWithCustomTypeValidator;
  var __super_addHeaderParam = this.addHeaderParam;
  var __super_addHeaderParamWithPattern = this.addHeaderParamWithPattern;
  var __super_addHeaderParamWithCustomTypeValidator = this.addHeaderParamWithCustomTypeValidator;
  var __super_addFormParam = this.addFormParam;
  var __super_addFormParamWithPattern = this.addFormParamWithPattern;
  var __super_addFormParamsArray = this.addFormParamsArray;
  var __super_addFormParamsArrayWithPattern = this.addFormParamsArrayWithPattern;
  var __super_addFormParamWithCustomTypeValidator = this.addFormParamWithCustomTypeValidator;
  var __super_addCustomValidatorFunction = this.addCustomValidatorFunction;
  var __super_addJsonBodySchema = this.addJsonBodySchema;
  var __super_addXMLBodySchema = this.addXMLBodySchema;
  var __super_addMultipartRequiredFile = this.addMultipartRequiredFile;
  var __super_addExpectedContentType = this.addExpectedContentType;
  /**
   Something has happened, so handle it.

   @public
   @param event {RoutingContext} the event to handle 
   */
  this.handle =  function(event) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_hTTPRequestValidationHandler["handle(io.vertx.ext.web.RoutingContext)"](event._jdel);
    } else if (typeof __super_handle != 'undefined') {
      return __super_handle.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a path parameter with included parameter types. All path params are <b>required</b>

   @public
   @param parameterName {string} expected name of parameter inside the path 
   @param type {Object} expected type of parameter 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addPathParam =  function(parameterName, type) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_hTTPRequestValidationHandler["addPathParam(java.lang.String,io.vertx.ext.web.api.validation.ParameterType)"](parameterName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type)) ;
      return that;
    } else if (typeof __super_addPathParam != 'undefined') {
      return __super_addPathParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a path parameter with a custom pattern. All path params are <b>required</b>

   @public
   @param parameterName {string} expected name of parameter inside the path 
   @param pattern {string} regular expression for validation 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addPathParamWithPattern =  function(parameterName, pattern) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_hTTPRequestValidationHandler["addPathParamWithPattern(java.lang.String,java.lang.String)"](parameterName, pattern) ;
      return that;
    } else if (typeof __super_addPathParamWithPattern != 'undefined') {
      return __super_addPathParamWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a path parameter with a custom type validator. All path params are <b>required</b>. For more informations
   about how to construct built-in or custom type validator, check out {@link ParameterTypeValidator}

   @public
   @param parameterName {string} expected name of parameter inside the path 
   @param validator {ParameterTypeValidator} type validator 
   @param allowEmptyValue {boolean} true if parameter allowEmptyValue. For more informations about allowEmptyValue behaviour: {@link ParameterValidationRule#allowEmptyValue} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addPathParamWithCustomTypeValidator =  function(parameterName, validator, allowEmptyValue) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addPathParamWithCustomTypeValidator(java.lang.String,io.vertx.ext.web.api.validation.ParameterTypeValidator,boolean)"](parameterName, validator._jdel, allowEmptyValue) ;
      return that;
    } else if (typeof __super_addPathParamWithCustomTypeValidator != 'undefined') {
      return __super_addPathParamWithCustomTypeValidator.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameter with included parameter types

   @public
   @param parameterName {string} expected name of parameter inside the query 
   @param type {Object} expected type of parameter 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addQueryParam =  function(parameterName, type, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addQueryParam(java.lang.String,io.vertx.ext.web.api.validation.ParameterType,boolean)"](parameterName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type), required) ;
      return that;
    } else if (typeof __super_addQueryParam != 'undefined') {
      return __super_addQueryParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameter with a custom pattern

   @public
   @param parameterName {string} expected name of parameter inside the query 
   @param pattern {string} regular expression for validation 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addQueryParamWithPattern =  function(parameterName, pattern, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addQueryParamWithPattern(java.lang.String,java.lang.String,boolean)"](parameterName, pattern, required) ;
      return that;
    } else if (typeof __super_addQueryParamWithPattern != 'undefined') {
      return __super_addQueryParamWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameters array with included parameter types

   @public
   @param arrayName {string} expected name of array inside the query 
   @param type {Object} expected type of parameter 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addQueryParamsArray =  function(arrayName, type, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addQueryParamsArray(java.lang.String,io.vertx.ext.web.api.validation.ParameterType,boolean)"](arrayName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type), required) ;
      return that;
    } else if (typeof __super_addQueryParamsArray != 'undefined') {
      return __super_addQueryParamsArray.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameters array with a custom pattern

   @public
   @param arrayName {string} expected name of array inside the query 
   @param pattern {string} regular expression for validation 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addQueryParamsArrayWithPattern =  function(arrayName, pattern, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addQueryParamsArrayWithPattern(java.lang.String,java.lang.String,boolean)"](arrayName, pattern, required) ;
      return that;
    } else if (typeof __super_addQueryParamsArrayWithPattern != 'undefined') {
      return __super_addQueryParamsArrayWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a query parameter with a custom type validator. For more informations about how to construct built-in or
   custom type validator, check out {@link ParameterTypeValidator}

   @public
   @param parameterName {string} expected name of parameter inside the query 
   @param validator {ParameterTypeValidator} type validator 
   @param required {boolean} true if parameter is required 
   @param allowEmptyValue {boolean} true if parameter allowEmptyValue. For more informations about allowEmptyValue behaviour: {@link ParameterValidationRule#allowEmptyValue} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addQueryParamWithCustomTypeValidator =  function(parameterName, validator, required, allowEmptyValue) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] ==='boolean' && typeof __args[3] ==='boolean') {
      j_hTTPRequestValidationHandler["addQueryParamWithCustomTypeValidator(java.lang.String,io.vertx.ext.web.api.validation.ParameterTypeValidator,boolean,boolean)"](parameterName, validator._jdel, required, allowEmptyValue) ;
      return that;
    } else if (typeof __super_addQueryParamWithCustomTypeValidator != 'undefined') {
      return __super_addQueryParamWithCustomTypeValidator.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a header parameter with included parameter types

   @public
   @param headerName {string} expected header name 
   @param type {Object} expected type of parameter 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addHeaderParam =  function(headerName, type, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addHeaderParam(java.lang.String,io.vertx.ext.web.api.validation.ParameterType,boolean)"](headerName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type), required) ;
      return that;
    } else if (typeof __super_addHeaderParam != 'undefined') {
      return __super_addHeaderParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a header parameter with a custom pattern

   @public
   @param headerName {string} expected header name 
   @param pattern {string} regular expression for validation 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addHeaderParamWithPattern =  function(headerName, pattern, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addHeaderParamWithPattern(java.lang.String,java.lang.String,boolean)"](headerName, pattern, required) ;
      return that;
    } else if (typeof __super_addHeaderParamWithPattern != 'undefined') {
      return __super_addHeaderParamWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a header parameter with a custom type validator. For more informations about how to construct built-in or
   custom type validator, check out {@link ParameterTypeValidator}

   @public
   @param headerName {string} expected header namery 
   @param validator {ParameterTypeValidator} type validator 
   @param required {boolean} true if parameter is required 
   @param allowEmptyValue {boolean} true if parameter allowEmptyValue. For more informations about allowEmptyValue behaviour: {@link ParameterValidationRule#allowEmptyValue} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addHeaderParamWithCustomTypeValidator =  function(headerName, validator, required, allowEmptyValue) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] ==='boolean' && typeof __args[3] ==='boolean') {
      j_hTTPRequestValidationHandler["addHeaderParamWithCustomTypeValidator(java.lang.String,io.vertx.ext.web.api.validation.ParameterTypeValidator,boolean,boolean)"](headerName, validator._jdel, required, allowEmptyValue) ;
      return that;
    } else if (typeof __super_addHeaderParamWithCustomTypeValidator != 'undefined') {
      return __super_addHeaderParamWithCustomTypeValidator.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a single parameter inside a form with included parameter types

   @public
   @param parameterName {string} expected name of parameter inside the form 
   @param type {Object} expected type of parameter 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addFormParam =  function(parameterName, type, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addFormParam(java.lang.String,io.vertx.ext.web.api.validation.ParameterType,boolean)"](parameterName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type), required) ;
      return that;
    } else if (typeof __super_addFormParam != 'undefined') {
      return __super_addFormParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a single parameter inside a form with a custom pattern

   @public
   @param parameterName {string} expected name of parameter inside the form 
   @param pattern {string} regular expression for validation 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addFormParamWithPattern =  function(parameterName, pattern, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addFormParamWithPattern(java.lang.String,java.lang.String,boolean)"](parameterName, pattern, required) ;
      return that;
    } else if (typeof __super_addFormParamWithPattern != 'undefined') {
      return __super_addFormParamWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a form parameters array with included parameter types

   @public
   @param parameterName {string} expected name of array of parameters inside the form 
   @param type {Object} expected type of array of parameters 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addFormParamsArray =  function(parameterName, type, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addFormParamsArray(java.lang.String,io.vertx.ext.web.api.validation.ParameterType,boolean)"](parameterName, io.vertx.ext.web.api.validation.ParameterType.valueOf(type), required) ;
      return that;
    } else if (typeof __super_addFormParamsArray != 'undefined') {
      return __super_addFormParamsArray.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a form parameters array with a custom pattern

   @public
   @param parameterName {string} expected name of array of parameters inside the form 
   @param pattern {string} regular expression for validation 
   @param required {boolean} true if parameter is required 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addFormParamsArrayWithPattern =  function(parameterName, pattern, required) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] ==='boolean') {
      j_hTTPRequestValidationHandler["addFormParamsArrayWithPattern(java.lang.String,java.lang.String,boolean)"](parameterName, pattern, required) ;
      return that;
    } else if (typeof __super_addFormParamsArrayWithPattern != 'undefined') {
      return __super_addFormParamsArrayWithPattern.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a form parameter with a custom type validator. For more informations about how to construct built-in or
   custom type validator, check out {@link ParameterTypeValidator}

   @public
   @param parameterName {string} expected name of parameter inside the form 
   @param validator {ParameterTypeValidator} type validator 
   @param required {boolean} true if parameter is required 
   @param allowEmptyValue {boolean} true if parameter allowEmptyValue. For more informations about allowEmptyValue behaviour: {@link ParameterValidationRule#allowEmptyValue} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addFormParamWithCustomTypeValidator =  function(parameterName, validator, required, allowEmptyValue) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] ==='boolean' && typeof __args[3] ==='boolean') {
      j_hTTPRequestValidationHandler["addFormParamWithCustomTypeValidator(java.lang.String,io.vertx.ext.web.api.validation.ParameterTypeValidator,boolean,boolean)"](parameterName, validator._jdel, required, allowEmptyValue) ;
      return that;
    } else if (typeof __super_addFormParamWithCustomTypeValidator != 'undefined') {
      return __super_addFormParamWithCustomTypeValidator.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a custom validator. For more informations about custom validator, see {@link CustomValidator}

   @public
   @param customValidator {CustomValidator} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addCustomValidatorFunction =  function(customValidator) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_hTTPRequestValidationHandler["addCustomValidatorFunction(io.vertx.ext.web.api.validation.CustomValidator)"](customValidator._jdel) ;
      return that;
    } else if (typeof __super_addCustomValidatorFunction != 'undefined') {
      return __super_addCustomValidatorFunction.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a json schema for body with Content-Type "application/json"

   @public
   @param jsonSchema {string} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addJsonBodySchema =  function(jsonSchema) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_hTTPRequestValidationHandler["addJsonBodySchema(java.lang.String)"](jsonSchema) ;
      return that;
    } else if (typeof __super_addJsonBodySchema != 'undefined') {
      return __super_addJsonBodySchema.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a xml schema for body with Content-Type "application/xml"

   @public
   @param xmlSchema {string} 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addXMLBodySchema =  function(xmlSchema) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_hTTPRequestValidationHandler["addXMLBodySchema(java.lang.String)"](xmlSchema) ;
      return that;
    } else if (typeof __super_addXMLBodySchema != 'undefined') {
      return __super_addXMLBodySchema.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add an expected filename inside <b>multipart request</b>.

   @public
   @param filename {string} name of the file inside the form 
   @param contentType {string} expected content type of file 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addMultipartRequiredFile =  function(filename, contentType) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_hTTPRequestValidationHandler["addMultipartRequiredFile(java.lang.String,java.lang.String)"](filename, contentType) ;
      return that;
    } else if (typeof __super_addMultipartRequiredFile != 'undefined') {
      return __super_addMultipartRequiredFile.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add an expected content type of request. It's not needed to add application/json, application/xml,
   multipart/form-data and application/x-www-form-urlencoded

   @public
   @param contentType {string} expected content type of file 
   @return {HTTPRequestValidationHandler} this handler
   */
  this.addExpectedContentType =  function(contentType) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_hTTPRequestValidationHandler["addExpectedContentType(java.lang.String)"](contentType) ;
      return that;
    } else if (typeof __super_addExpectedContentType != 'undefined') {
      return __super_addExpectedContentType.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_hTTPRequestValidationHandler;
};

HTTPRequestValidationHandler._jclass = utils.getJavaClass("io.vertx.ext.web.api.validation.HTTPRequestValidationHandler");
HTTPRequestValidationHandler._jtype = {accept: function(obj) {
    return HTTPRequestValidationHandler._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(HTTPRequestValidationHandler.prototype, {});
    HTTPRequestValidationHandler.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
HTTPRequestValidationHandler._create = function(jdel) {var obj = Object.create(HTTPRequestValidationHandler.prototype, {});
  HTTPRequestValidationHandler.apply(obj, arguments);
  return obj;
}
/**
 Factory method to create an HTTPRequestValidationHandler

 @memberof module:vertx-web-api-contract-js/http_request_validation_handler

 @return {HTTPRequestValidationHandler} new HTTPRequestValidationHandler
 */
HTTPRequestValidationHandler.create =  function() {
  var __args = arguments;
  if (__args.length === 0) {
    return utils.convReturnVertxGen(HTTPRequestValidationHandler, JHTTPRequestValidationHandler["create()"]()) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = HTTPRequestValidationHandler;