/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/record_parser */
var utils = require('vertx-js/util/utils');
var Pipe = require('vertx-js/pipe');
var Buffer = require('vertx-js/buffer');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRecordParser = Java.type('io.vertx.core.parsetools.RecordParser');

/**
 A helper class which allows you to easily parse protocols which are delimited by a sequence of bytes, or fixed
 size records.
 <p>
 @class
*/
var RecordParser = function(j_val) {

  var j_recordParser = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_pipe = this.pipe;
  var __super_pipeTo = this.pipeTo;
  var __super_setOutput = this.setOutput;
  var __super_newDelimited = this.newDelimited;
  var __super_newDelimited = this.newDelimited;
  var __super_newDelimited = this.newDelimited;
  var __super_newDelimited = this.newDelimited;
  var __super_newDelimited = this.newDelimited;
  var __super_newDelimited = this.newDelimited;
  var __super_newFixed = this.newFixed;
  var __super_newFixed = this.newFixed;
  var __super_newFixed = this.newFixed;
  var __super_delimitedMode = this.delimitedMode;
  var __super_delimitedMode = this.delimitedMode;
  var __super_fixedSizeMode = this.fixedSizeMode;
  var __super_maxRecordSize = this.maxRecordSize;
  var __super_handle = this.handle;
  var __super_exceptionHandler = this.exceptionHandler;
  var __super_handler = this.handler;
  var __super_pause = this.pause;
  var __super_fetch = this.fetch;
  var __super_resume = this.resume;
  var __super_endHandler = this.endHandler;
  /**
   Pause this stream and return a  to transfer the elements of this stream to a destination .
   <p/>
   The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.

   @public

   @return {Pipe} a pipe
   */
  this.pipe =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Pipe, j_recordParser["pipe()"](), Buffer._jtype) ;
    } else if (typeof __super_pipe != 'undefined') {
      return __super_pipe.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   <p>
   Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   <p>
   Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   called with the result.

   @public
   @param dst {WriteStream} the destination write stream 
   @param handler {function} 
   */
  this.pipeTo =  function(dst, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_recordParser["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_recordParser["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_pipeTo != 'undefined') {
      return __super_pipeTo.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param output {function} 
   */
  this.setOutput =  function(output) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_recordParser["setOutput(io.vertx.core.Handler)"](function(jVal) {
        output(utils.convReturnVertxGen(Buffer, jVal));
      });
    } else if (typeof __super_setOutput != 'undefined') {
      return __super_setOutput.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser into delimited mode, and where the delimiter can be represented
   by the delimiter <code>delim</code>.
   <p>
   This method can be called multiple times with different values of delim while data is being parsed.

   @public
   @param delim {Buffer} the new delimiter 
   */
  this.delimitedMode =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_recordParser["delimitedMode(java.lang.String)"](__args[0]);
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_recordParser["delimitedMode(io.vertx.core.buffer.Buffer)"](__args[0]._jdel);
    } else if (typeof __super_delimitedMode != 'undefined') {
      return __super_delimitedMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser into fixed size mode, where the record size is specified by <code>size</code> in bytes.
   <p>
   This method can be called multiple times with different values of size while data is being parsed.

   @public
   @param size {number} the new record size 
   */
  this.fixedSizeMode =  function(size) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_recordParser["fixedSizeMode(int)"](size);
    } else if (typeof __super_fixedSizeMode != 'undefined') {
      return __super_fixedSizeMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the maximum allowed size for a record when using the delimited mode.
   The delimiter itself does not count for the record size.
   <p>
   If a record is longer than specified, an IllegalStateException will be thrown.

   @public
   @param size {number} the maximum record size 
   @return {RecordParser} a reference to this, so the API can be used fluently
   */
  this.maxRecordSize =  function(size) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_recordParser["maxRecordSize(int)"](size) ;
      return that;
    } else if (typeof __super_maxRecordSize != 'undefined') {
      return __super_maxRecordSize.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   This method is called to provide the parser with data.

   @public
   @param buffer {Buffer} a chunk of data 
   */
  this.handle =  function(buffer) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_recordParser["handle(io.vertx.core.buffer.Buffer)"](buffer._jdel);
    } else if (typeof __super_handle != 'undefined') {
      return __super_handle.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param handler {function} 
   @return {RecordParser}
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_recordParser["exceptionHandler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param handler {function} 
   @return {RecordParser}
   */
  this.handler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_recordParser["handler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnVertxGen(Buffer, jVal));
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {RecordParser}
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_recordParser["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param amount {number} 
   @return {RecordParser}
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_recordParser["fetch(long)"](amount) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {RecordParser}
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_recordParser["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param endHandler {function} 
   @return {RecordParser}
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_recordParser["endHandler(io.vertx.core.Handler)"](endHandler) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_recordParser;
};

RecordParser._jclass = utils.getJavaClass("io.vertx.core.parsetools.RecordParser");
RecordParser._jtype = {accept: function(obj) {
    return RecordParser._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(RecordParser.prototype, {});
    RecordParser.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
RecordParser._create = function(jdel) {var obj = Object.create(RecordParser.prototype, {});
  RecordParser.apply(obj, arguments);
  return obj;
}
/**
 Like {@link RecordParser#newDelimited} but wraps the <code>stream</code>. The <code>stream</code> handlers will be set/unset
 when the {@link RecordParser#handler} is set.
 <p/>
 The <code>pause()</code>/<code>resume()</code> operations are propagated to the <code>stream</code>.

 @memberof module:vertx-js/record_parser
 @param delim {Buffer} the initial delimiter buffer 
 @param stream {ReadStream} the wrapped stream 
 @return {RecordParser}
 */
RecordParser.newDelimited =  function() {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(java.lang.String,io.vertx.core.Handler)"](__args[0], function(jVal) {
      __args[1](utils.convReturnVertxGen(Buffer, jVal));
    })) ;
  } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1]._jdel) {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(java.lang.String,io.vertx.core.streams.ReadStream)"](__args[0], __args[1]._jdel)) ;
  } else if (__args.length === 1 && typeof __args[0] === 'string') {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(java.lang.String)"](__args[0])) ;
  } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(io.vertx.core.buffer.Buffer)"](__args[0]._jdel)) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(io.vertx.core.buffer.Buffer,io.vertx.core.Handler)"](__args[0]._jdel, function(jVal) {
      __args[1](utils.convReturnVertxGen(Buffer, jVal));
    })) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'object' && __args[1]._jdel) {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newDelimited(io.vertx.core.buffer.Buffer,io.vertx.core.streams.ReadStream)"](__args[0]._jdel, __args[1]._jdel)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

/**
 Like {@link RecordParser#newFixed} but wraps the <code>stream</code>. The <code>stream</code> handlers will be set/unset
 when the {@link RecordParser#handler} is set.
 <p/>
 The <code>pause()</code>/<code>resume()</code> operations are propagated to the <code>stream</code>.

 @memberof module:vertx-js/record_parser
 @param size {number} the initial record size 
 @param stream {ReadStream} the wrapped stream 
 @return {RecordParser}
 */
RecordParser.newFixed =  function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] ==='number') {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newFixed(int)"](__args[0])) ;
  } else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'function') {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newFixed(int,io.vertx.core.Handler)"](__args[0], function(jVal) {
      __args[1](utils.convReturnVertxGen(Buffer, jVal));
    })) ;
  } else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'object' && __args[1]._jdel) {
    return utils.convReturnVertxGen(RecordParser, JRecordParser["newFixed(int,io.vertx.core.streams.ReadStream)"](__args[0], __args[1]._jdel)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = RecordParser;