/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-pg-client-js/pg_channel */
var utils = require('vertx-js/util/utils');
var Pipe = require('vertx-js/pipe');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JPgChannel = Java.type('io.vertx.pgclient.pubsub.PgChannel');

/**

 @class
*/
var PgChannel = function(j_val) {

  var j_pgChannel = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_fetch = this.fetch;
  var __super_pipe = this.pipe;
  var __super_pipeTo = this.pipeTo;
  var __super_subscribeHandler = this.subscribeHandler;
  var __super_handler = this.handler;
  var __super_pause = this.pause;
  var __super_resume = this.resume;
  var __super_endHandler = this.endHandler;
  var __super_exceptionHandler = this.exceptionHandler;
  /**
   Fetch the specified <code>amount</code> of elements. If the <code>ReadStream</code> has been paused, reading will
   recommence with the specified <code>amount</code> of items, otherwise the specified <code>amount</code> will
   be added to the current stream demand.

   @public
   @param amount {number} 
   @return {ReadStream} a reference to this, so the API can be used fluently
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_pgChannel["fetch(long)"](amount) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pause this stream and return a  to transfer the elements of this stream to a destination .
   <p/>
   The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.

   @public

   @return {Pipe} a pipe
   */
  this.pipe =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Pipe, j_pgChannel["pipe()"](), undefined) ;
    } else if (typeof __super_pipe != 'undefined') {
      return __super_pipe.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   <p>
   Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   <p>
   Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   called with the result.

   @public
   @param dst {WriteStream} the destination write stream 
   @param handler {function} 
   */
  this.pipeTo =  function(dst, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_pgChannel["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_pgChannel["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_pipeTo != 'undefined') {
      return __super_pipeTo.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an handler called when the the channel get subscribed.

   @public
   @param handler {function} the handler 
   @return {PgChannel} a reference to this, so the API can be used fluently
   */
  this.subscribeHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_pgChannel["subscribeHandler(io.vertx.core.Handler)"](handler) ;
      return that;
    } else if (typeof __super_subscribeHandler != 'undefined') {
      return __super_subscribeHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set or unset an handler to be called when a the channel is notified by Postgres.
   <p/>
   <ul>
     <li>when the handler is set, the subscriber sends a <code>LISTEN</code> command if needed</li>
     <li>when the handler is unset, the subscriber sends a <code>UNLISTEN</code> command if needed</li>
   </ul>

   @public
   @param handler {function} the handler 
   @return {PgChannel} a reference to this, so the API can be used fluently
   */
  this.handler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_pgChannel["handler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(jVal);
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pause the channel, all notifications are discarded.

   @public

   @return {PgChannel} a reference to this, so the API can be used fluently
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_pgChannel["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Resume the channel.

   @public

   @return {PgChannel} a reference to this, so the API can be used fluently
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_pgChannel["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set an handler to be called when no more notifications will be received.

   @public
   @param endHandler {function} the handler 
   @return {PgChannel} a reference to this, so the API can be used fluently
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_pgChannel["endHandler(io.vertx.core.Handler)"](endHandler) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param handler {function} 
   @return {PgChannel}
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_pgChannel["exceptionHandler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_pgChannel;
};

PgChannel._jclass = utils.getJavaClass("io.vertx.pgclient.pubsub.PgChannel");
PgChannel._jtype = {accept: function(obj) {
    return PgChannel._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(PgChannel.prototype, {});
    PgChannel.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
PgChannel._create = function(jdel) {var obj = Object.create(PgChannel.prototype, {});
  PgChannel.apply(obj, arguments);
  return obj;
}
module.exports = PgChannel;