/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/json_parser */
var utils = require('vertx-js/util/utils');
var Pipe = require('vertx-js/pipe');
var Buffer = require('vertx-js/buffer');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');
var JsonEvent = require('vertx-js/json_event');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JJsonParser = Java.type('io.vertx.core.parsetools.JsonParser');

/**
 A parser class which allows to incrementally parse json elements and emit json parse events instead of parsing a json
 element fully. This parser is convenient for parsing large json structures.
 <p/>
 The parser also parses concatenated json streams or line delimited json streams.
 <p/>
 The parser can also parse entire object or array when it is convenient, for instance a very large array
 of small objects can be parsed efficiently by handling array <i>start</i>/<i>end</i> and <i>object</i>
 events.
 <p/>
 @class
*/
var JsonParser = function(j_val) {

  var j_jsonParser = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_handle = this.handle;
  var __super_pipe = this.pipe;
  var __super_pipeTo = this.pipeTo;
  var __super_newParser = this.newParser;
  var __super_newParser = this.newParser;
  var __super_write = this.write;
  var __super_end = this.end;
  var __super_objectEventMode = this.objectEventMode;
  var __super_objectValueMode = this.objectValueMode;
  var __super_arrayEventMode = this.arrayEventMode;
  var __super_arrayValueMode = this.arrayValueMode;
  var __super_pause = this.pause;
  var __super_resume = this.resume;
  var __super_fetch = this.fetch;
  var __super_endHandler = this.endHandler;
  var __super_handler = this.handler;
  var __super_exceptionHandler = this.exceptionHandler;
  /**
   Something has happened, so handle it.

   @public
   @param event {Buffer} the event to handle 
   */
  this.handle =  function(event) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_jsonParser["handle(io.vertx.core.buffer.Buffer)"](event._jdel);
    } else if (typeof __super_handle != 'undefined') {
      return __super_handle.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pause this stream and return a  to transfer the elements of this stream to a destination .
   <p/>
   The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.

   @public

   @return {Pipe} a pipe
   */
  this.pipe =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Pipe, j_jsonParser["pipe()"](), JsonEvent._jtype) ;
    } else if (typeof __super_pipe != 'undefined') {
      return __super_pipe.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   <p>
   Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   <p>
   Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   called with the result.

   @public
   @param dst {WriteStream} the destination write stream 
   @param handler {function} 
   */
  this.pipeTo =  function(dst, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_jsonParser["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_jsonParser["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_pipeTo != 'undefined') {
      return __super_pipeTo.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Handle a <code>Buffer</code>, pretty much like calling {@link Handler#handle}.

   @public
   @param buffer {Buffer} 
   @return {JsonParser} a reference to this, so the API can be used fluently
   */
  this.write =  function(buffer) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_jsonParser["write(io.vertx.core.buffer.Buffer)"](buffer._jdel) ;
      return that;
    } else if (typeof __super_write != 'undefined') {
      return __super_write.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   End the stream, this must be called after all the json stream has been processed.

   @public

   */
  this.end =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["end()"]();
    } else if (typeof __super_end != 'undefined') {
      return __super_end.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser to emit a stream of events for each new json object.

   @public

   @return {JsonParser} a reference to this, so the API can be used fluently
   */
  this.objectEventMode =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["objectEventMode()"]() ;
      return that;
    } else if (typeof __super_objectEventMode != 'undefined') {
      return __super_objectEventMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser to emit a single value event for each new json object.
   </p>
   Json object currently streamed won't be affected.

   @public

   @return {JsonParser} a reference to this, so the API can be used fluently
   */
  this.objectValueMode =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["objectValueMode()"]() ;
      return that;
    } else if (typeof __super_objectValueMode != 'undefined') {
      return __super_objectValueMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser to emit a stream of events for each new json array.

   @public

   @return {JsonParser} a reference to this, so the API can be used fluently
   */
  this.arrayEventMode =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["arrayEventMode()"]() ;
      return that;
    } else if (typeof __super_arrayEventMode != 'undefined') {
      return __super_arrayEventMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flip the parser to emit a single value event for each new json array.
   </p>
   Json array currently streamed won't be affected.

   @public

   @return {JsonParser} a reference to this, so the API can be used fluently
   */
  this.arrayValueMode =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["arrayValueMode()"]() ;
      return that;
    } else if (typeof __super_arrayValueMode != 'undefined') {
      return __super_arrayValueMode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {JsonParser}
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {JsonParser}
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonParser["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param amount {number} 
   @return {JsonParser}
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_jsonParser["fetch(long)"](amount) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param endHandler {function} 
   @return {JsonParser}
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_jsonParser["endHandler(io.vertx.core.Handler)"](endHandler) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param handler {function} 
   @return {JsonParser}
   */
  this.handler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_jsonParser["handler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnVertxGen(JsonEvent, jVal));
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param handler {function} 
   @return {JsonParser}
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_jsonParser["exceptionHandler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_jsonParser;
};

JsonParser._jclass = utils.getJavaClass("io.vertx.core.parsetools.JsonParser");
JsonParser._jtype = {accept: function(obj) {
    return JsonParser._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(JsonParser.prototype, {});
    JsonParser.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
JsonParser._create = function(jdel) {var obj = Object.create(JsonParser.prototype, {});
  JsonParser.apply(obj, arguments);
  return obj;
}
/**
 Create a new <code>JsonParser</code> instance.

 @memberof module:vertx-js/json_parser
 @param stream {ReadStream} 
 @return {JsonParser}
 */
JsonParser.newParser =  function() {
  var __args = arguments;
  if (__args.length === 0) {
    return utils.convReturnVertxGen(JsonParser, JJsonParser["newParser()"]()) ;
  } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(JsonParser, JJsonParser["newParser(io.vertx.core.streams.ReadStream)"](__args[0]._jdel)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = JsonParser;