/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-mongo-js/mongo_grid_fs_client */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JMongoGridFsClient = Java.type('io.vertx.ext.mongo.MongoGridFsClient');
var GridFsDownloadOptions = Java.type('io.vertx.ext.mongo.GridFsDownloadOptions');
var GridFsUploadOptions = Java.type('io.vertx.ext.mongo.GridFsUploadOptions');

/**
 @class
*/
var MongoGridFsClient = function(j_val) {

  var j_mongoGridFsClient = j_val;
  var that = this;

  var __super_delete = this.delete;
  var __super_downloadByFileName = this.downloadByFileName;
  var __super_downloadByFileNameWithOptions = this.downloadByFileNameWithOptions;
  var __super_downloadById = this.downloadById;
  var __super_downloadFile = this.downloadFile;
  var __super_downloadFileAs = this.downloadFileAs;
  var __super_downloadFileByID = this.downloadFileByID;
  var __super_drop = this.drop;
  var __super_findAllIds = this.findAllIds;
  var __super_findIds = this.findIds;
  var __super_uploadByFileName = this.uploadByFileName;
  var __super_uploadByFileNameWithOptions = this.uploadByFileNameWithOptions;
  var __super_uploadFile = this.uploadFile;
  var __super_uploadFileWithOptions = this.uploadFileWithOptions;
  var __super_close = this.close;
  /**
   Deletes a file by it's ID

   @public
   @param id {string} the identifier of the file 
   @param resultHandler {function} will be called when the file is deleted 
   @return {MongoGridFsClient}
   */
  this.delete =  function(id, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoGridFsClient["delete(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["delete(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_delete != 'undefined') {
      return __super_delete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param stream {WriteStream} 
   @param fileName {string} 
   @param resultHandler {function} 
   @return {MongoGridFsClient}
   */
  this.downloadByFileName =  function(stream, fileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoGridFsClient["downloadByFileName(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, fileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadByFileName(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, fileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadByFileName != 'undefined') {
      return __super_downloadByFileName.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param stream {WriteStream} 
   @param fileName {string} 
   @param options {Object} 
   @param resultHandler {function} 
   @return {MongoGridFsClient}
   */
  this.downloadByFileNameWithOptions =  function(stream, fileName, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoGridFsClient["downloadByFileNameWithOptions(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.ext.mongo.GridFsDownloadOptions,io.vertx.core.Handler)"](stream._jdel, fileName, options  != null ? new GridFsDownloadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadByFileNameWithOptions(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.ext.mongo.GridFsDownloadOptions,io.vertx.core.Handler)"](stream._jdel, fileName, options  != null ? new GridFsDownloadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadByFileNameWithOptions != 'undefined') {
      return __super_downloadByFileNameWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param stream {WriteStream} 
   @param id {string} 
   @param resultHandler {function} 
   @return {MongoGridFsClient}
   */
  this.downloadById =  function(stream, id, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoGridFsClient["downloadById(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, id, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadById(io.vertx.core.streams.WriteStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, id, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadById != 'undefined') {
      return __super_downloadById.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Downloads a file.

   @public
   @param fileName {string} the name of the file to download 
   @param resultHandler {function} called when the file is downloaded and returns the length in bytes 
   @return {MongoGridFsClient}
   */
  this.downloadFile =  function(fileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoGridFsClient["downloadFile(java.lang.String,io.vertx.core.Handler)"](fileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadFile(java.lang.String,io.vertx.core.Handler)"](fileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadFile != 'undefined') {
      return __super_downloadFile.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Downloads a file and gives it a new name.

   @public
   @param fileName {string} the name of the file to download 
   @param newFileName {string} the name the file should be saved as 
   @param resultHandler {function} called when the file is downloaded and returns the length in bytes 
   @return {MongoGridFsClient}
   */
  this.downloadFileAs =  function(fileName, newFileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoGridFsClient["downloadFileAs(java.lang.String,java.lang.String,io.vertx.core.Handler)"](fileName, newFileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadFileAs(java.lang.String,java.lang.String,io.vertx.core.Handler)"](fileName, newFileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadFileAs != 'undefined') {
      return __super_downloadFileAs.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Downloads a file using the ID generated by GridFs.

   @public
   @param id {string} the GridFs Object ID of the file to download 
   @param fileName {string} 
   @param resultHandler {function} called when the file is downloaded and returns the length in bytes 
   @return {MongoGridFsClient}
   */
  this.downloadFileByID =  function(id, fileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoGridFsClient["downloadFileByID(java.lang.String,java.lang.String,io.vertx.core.Handler)"](id, fileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["downloadFileByID(java.lang.String,java.lang.String,io.vertx.core.Handler)"](id, fileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_downloadFileByID != 'undefined') {
      return __super_downloadFileByID.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Drops the entire file bucket with all of its contents

   @public
   @param resultHandler {function} called when the bucket is dropped 
   @return {MongoGridFsClient}
   */
  this.drop =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mongoGridFsClient["drop(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["drop(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_drop != 'undefined') {
      return __super_drop.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Finds all file ids in the bucket

   @public
   @param resultHandler {function} called when the list of file ids is available 
   @return {MongoGridFsClient}
   */
  this.findAllIds =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mongoGridFsClient["findAllIds(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["findAllIds(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findAllIds != 'undefined') {
      return __super_findAllIds.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Finds all file ids that match a query.

   @public
   @param query {Object} a bson query expressed as json that will be used to match files 
   @param resultHandler {function} called when the list of file ids is available 
   @return {MongoGridFsClient}
   */
  this.findIds =  function(query, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && (typeof __args[0] === 'object' && __args[0] != null) && typeof __args[1] === 'function') {
      j_mongoGridFsClient["findIds(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["findIds(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findIds != 'undefined') {
      return __super_findIds.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param stream {ReadStream} 
   @param fileName {string} 
   @param resultHandler {function} 
   @return {MongoGridFsClient}
   */
  this.uploadByFileName =  function(stream, fileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoGridFsClient["uploadByFileName(io.vertx.core.streams.ReadStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, fileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["uploadByFileName(io.vertx.core.streams.ReadStream,java.lang.String,io.vertx.core.Handler)"](stream._jdel, fileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_uploadByFileName != 'undefined') {
      return __super_uploadByFileName.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public
   @param stream {ReadStream} 
   @param fileName {string} 
   @param options {Object} 
   @param resultHandler {function} 
   @return {MongoGridFsClient}
   */
  this.uploadByFileNameWithOptions =  function(stream, fileName, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoGridFsClient["uploadByFileNameWithOptions(io.vertx.core.streams.ReadStream,java.lang.String,io.vertx.ext.mongo.GridFsUploadOptions,io.vertx.core.Handler)"](stream._jdel, fileName, options  != null ? new GridFsUploadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["uploadByFileNameWithOptions(io.vertx.core.streams.ReadStream,java.lang.String,io.vertx.ext.mongo.GridFsUploadOptions,io.vertx.core.Handler)"](stream._jdel, fileName, options  != null ? new GridFsUploadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_uploadByFileNameWithOptions != 'undefined') {
      return __super_uploadByFileNameWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Upload a file to gridfs

   @public
   @param fileName {string} the name of the file to store in gridfs 
   @param resultHandler {function} the id of the file that was uploaded 
   @return {MongoGridFsClient}
   */
  this.uploadFile =  function(fileName, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoGridFsClient["uploadFile(java.lang.String,io.vertx.core.Handler)"](fileName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["uploadFile(java.lang.String,io.vertx.core.Handler)"](fileName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_uploadFile != 'undefined') {
      return __super_uploadFile.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Upload a file to gridfs with options

   @public
   @param fileName {string} the name of the file to store in gridfs 
   @param options {Object} <a href="../../dataobjects.html#GridFsUploadOptions">GridFsUploadOptions</a> for specifying metadata and chunk size 
   @param resultHandler {function} the id of the file that was uploaded 
   @return {MongoGridFsClient}
   */
  this.uploadFileWithOptions =  function(fileName, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoGridFsClient["uploadFileWithOptions(java.lang.String,io.vertx.ext.mongo.GridFsUploadOptions,io.vertx.core.Handler)"](fileName, options  != null ? new GridFsUploadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoGridFsClient["uploadFileWithOptions(java.lang.String,io.vertx.ext.mongo.GridFsUploadOptions,io.vertx.core.Handler)"](fileName, options  != null ? new GridFsUploadOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_uploadFileWithOptions != 'undefined') {
      return __super_uploadFileWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Close the client and release its resources

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_mongoGridFsClient["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_mongoGridFsClient;
};

MongoGridFsClient._jclass = utils.getJavaClass("io.vertx.ext.mongo.MongoGridFsClient");
MongoGridFsClient._jtype = {accept: function(obj) {
    return MongoGridFsClient._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(MongoGridFsClient.prototype, {});
    MongoGridFsClient.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
MongoGridFsClient._create = function(jdel) {var obj = Object.create(MongoGridFsClient.prototype, {});
  MongoGridFsClient.apply(obj, arguments);
  return obj;
}
module.exports = MongoGridFsClient;