/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-sql-client-js/row */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');
var Tuple = require('vertx-sql-client-js/tuple');
var Promise = require('vertx-js/promise');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRow = Java.type('io.vertx.sqlclient.Row');

/**
 Represents single row of the result set.

 @class
*/
var Row = function(j_val) {

  var j_row = j_val;
  var that = this;
  Tuple.call(this, j_val);

  var __super_getColumnName = this.getColumnName;
  var __super_getColumnIndex = this.getColumnIndex;
  var __super_getBoolean = this.getBoolean;
  var __super_getValue = this.getValue;
  var __super_getShort = this.getShort;
  var __super_getInteger = this.getInteger;
  var __super_getLong = this.getLong;
  var __super_getFloat = this.getFloat;
  var __super_getDouble = this.getDouble;
  var __super_getString = this.getString;
  var __super_getBuffer = this.getBuffer;
  /**
   Get a column name at <code>pos</code>.

   @public
   @param pos {number} the column position 
   @return {string} the column name or <code>null</code>
   */
  this.getColumnName =  function(pos) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_row["getColumnName(int)"](pos) ;
    } else if (typeof __super_getColumnName != 'undefined') {
      return __super_getColumnName.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a column position for the given column <code>name</code>.

   @public
   @param name {string} the column name 
   @return {number} the column name or <code>-1</code> if not found
   */
  this.getColumnIndex =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getColumnIndex(java.lang.String)"](name) ;
    } else if (typeof __super_getColumnIndex != 'undefined') {
      return __super_getColumnIndex.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a boolean value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {boolean} the value or <code>null</code>
   */
  this.getBoolean =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getBoolean(java.lang.String)"](name) ;
    } else if (typeof __super_getBoolean != 'undefined') {
      return __super_getBoolean.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get an object value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {Object} the value or <code>null</code>
   */
  this.getValue =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnTypeUnknown(j_row["getValue(java.lang.String)"](name)) ;
    } else if (typeof __super_getValue != 'undefined') {
      return __super_getValue.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a short value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {number} the value or <code>null</code>
   */
  this.getShort =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getShort(java.lang.String)"](name) ;
    } else if (typeof __super_getShort != 'undefined') {
      return __super_getShort.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get an integer value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {number} the value or <code>null</code>
   */
  this.getInteger =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getInteger(java.lang.String)"](name) ;
    } else if (typeof __super_getInteger != 'undefined') {
      return __super_getInteger.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a long value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {number} the value or <code>null</code>
   */
  this.getLong =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnLong(j_row["getLong(java.lang.String)"](name)) ;
    } else if (typeof __super_getLong != 'undefined') {
      return __super_getLong.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a float value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {number} the value or <code>null</code>
   */
  this.getFloat =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getFloat(java.lang.String)"](name) ;
    } else if (typeof __super_getFloat != 'undefined') {
      return __super_getFloat.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a double value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {number} the value or <code>null</code>
   */
  this.getDouble =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getDouble(java.lang.String)"](name) ;
    } else if (typeof __super_getDouble != 'undefined') {
      return __super_getDouble.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a string value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {string} the value or <code>null</code>
   */
  this.getString =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_row["getString(java.lang.String)"](name) ;
    } else if (typeof __super_getString != 'undefined') {
      return __super_getString.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a buffer value at <code>pos</code>.

   @public
   @param name {string} the column 
   @return {Buffer} the value or <code>null</code>
   */
  this.getBuffer =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Buffer, j_row["getBuffer(java.lang.String)"](name)) ;
    } else if (typeof __super_getBuffer != 'undefined') {
      return __super_getBuffer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_row;
};

Row._jclass = utils.getJavaClass("io.vertx.sqlclient.Row");
Row._jtype = {accept: function(obj) {
    return Row._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Row.prototype, {});
    Row.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Row._create = function(jdel) {var obj = Object.create(Row.prototype, {});
  Row.apply(obj, arguments);
  return obj;
}
Row.JSON_NULL = utils.convReturnTypeUnknown(JRow.JSON_NULL);
module.exports = Row;