/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-auth-common-js/auth_store */
var utils = require('vertx-js/util/utils');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JAuthStore = Java.type('io.vertx.ext.auth.AuthStore');

/**
 Generic interface to fetch user related information from a server backend.

 All methods of this interface are optional.

 @class
*/
var AuthStore = function(j_val) {

  var j_authStore = j_val;
  var that = this;

  var __super_generateId = this.generateId;
  var __super_getUserCredentialsByName = this.getUserCredentialsByName;
  var __super_getUserCredentialsById = this.getUserCredentialsById;
  var __super_updateUserCredential = this.updateUserCredential;
  var __super_getUserRoles = this.getUserRoles;
  var __super_getUserPermissions = this.getUserPermissions;
  /**
   Generates a unique ID that doesn't contain any user identifiable information. By default it generates a random UUID.
   Although this is will work for most cases a better implementation that prevents collisions of IDs should be
   preferred.

   @public

   @return {string} a new unique ID as a string
   */
  this.generateId =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_authStore["generateId()"]() ;
    } else if (typeof __super_generateId != 'undefined') {
      return __super_generateId.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Retrieves the user credentials from a backend given the user unique identifier.
   It may return more than 1 result, for example when a user can be identified using different modalities.

   @public
   @param username {string} user unique name. 
   @param handler {function} the handler for the result callback. 
   @return {AuthStore} fluent self.
   */
  this.getUserCredentialsByName =  function(username, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_authStore["getUserCredentialsByName(java.lang.String,io.vertx.core.Handler)"](username, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authStore["getUserCredentialsByName(java.lang.String,io.vertx.core.Handler)"](username, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_getUserCredentialsByName != 'undefined') {
      return __super_getUserCredentialsByName.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Retrieves the user credentials from a backend given the user unique identifier.
   It may return more than 1 result, for example when a user can be identified using different modalities.

   @public
   @param rawId {string} user unique rawId. 
   @param handler {function} the handler for the result callback. 
   @return {AuthStore} fluent self.
   */
  this.getUserCredentialsById =  function(rawId, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_authStore["getUserCredentialsById(java.lang.String,io.vertx.core.Handler)"](rawId, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authStore["getUserCredentialsById(java.lang.String,io.vertx.core.Handler)"](rawId, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_getUserCredentialsById != 'undefined') {
      return __super_getUserCredentialsById.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Update the user credential.

   @public
   @param id {string} the unique user identifier. 
   @param data {Object} the data to update. 
   @param upsert {boolean} insert if not present. 
   @param handler {function} the handler for the result callback. 
   @return {AuthStore} fluent self.
   */
  this.updateUserCredential =  function(id, data, upsert, handler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] ==='boolean' && typeof __args[3] === 'function') {
      j_authStore["updateUserCredential(java.lang.String,io.vertx.core.json.JsonObject,boolean,io.vertx.core.Handler)"](id, utils.convParamJsonObject(data), upsert, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] ==='boolean') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authStore["updateUserCredential(java.lang.String,io.vertx.core.json.JsonObject,boolean,io.vertx.core.Handler)"](id, utils.convParamJsonObject(data), upsert, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_updateUserCredential != 'undefined') {
      return __super_updateUserCredential.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the user roles from the storage.

   @public
   @param id {string} the unique user identifier. 
   @param handler {function} the handler for the result callback. 
   @return {AuthStore} fluent self.
   */
  this.getUserRoles =  function(id, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_authStore["getUserRoles(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          handler(ar.result(), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authStore["getUserRoles(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_getUserRoles != 'undefined') {
      return __super_getUserRoles.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the user permissions from the storage.

   @public
   @param id {string} the unique user identifier. 
   @param handler {function} the handler for the result callback. 
   @return {AuthStore} fluent self.
   */
  this.getUserPermissions =  function(id, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_authStore["getUserPermissions(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          handler(ar.result(), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authStore["getUserPermissions(java.lang.String,io.vertx.core.Handler)"](id, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_getUserPermissions != 'undefined') {
      return __super_getUserPermissions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_authStore;
};

AuthStore._jclass = utils.getJavaClass("io.vertx.ext.auth.AuthStore");
AuthStore._jtype = {accept: function(obj) {
    return AuthStore._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(AuthStore.prototype, {});
    AuthStore.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
AuthStore._create = function(jdel) {var obj = Object.create(AuthStore.prototype, {});
  AuthStore.apply(obj, arguments);
  return obj;
}
module.exports = AuthStore;
var Promise = require('vertx-js/promise');
