/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-auth-common-js/authentication_provider */
var utils = require('vertx-js/util/utils');
var User = require('vertx-auth-common-js/user');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JAuthenticationProvider = Java.type('io.vertx.ext.auth.authentication.AuthenticationProvider');

/**

 User-facing interface for authenticating users.

 @class
*/
var AuthenticationProvider = function(j_val) {

  var j_authenticationProvider = j_val;
  var that = this;

  var __super_authenticate = this.authenticate;
  /**
   Authenticate a user.
   <p>
   The first argument is a JSON object containing information for authenticating the user. What this actually contains
   depends on the specific implementation. In the case of a simple username/password based
   authentication it is likely to contain a JSON object with the following structure:
   <pre>
     {
       "username": "tim",
       "password": "mypassword"
     }
   </pre>
   For other types of authentication it contain different information - for example a JWT token or OAuth bearer token.
   <p>
   If the user is successfully authenticated a  object is passed to the handler in an .
   The user object can then be used for authorisation.

   @public
   @param authInfo {Object} The auth information 
   @param resultHandler {function} The result handler 
   */
  this.authenticate =  function(authInfo, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && (typeof __args[0] === 'object' && __args[0] != null) && typeof __args[1] === 'function') {
      j_authenticationProvider["authenticate(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(authInfo), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnVertxGen(User, ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_authenticationProvider["authenticate(io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](utils.convParamJsonObject(authInfo), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(User, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_authenticate != 'undefined') {
      return __super_authenticate.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_authenticationProvider;
};

AuthenticationProvider._jclass = utils.getJavaClass("io.vertx.ext.auth.authentication.AuthenticationProvider");
AuthenticationProvider._jtype = {accept: function(obj) {
    return AuthenticationProvider._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(AuthenticationProvider.prototype, {});
    AuthenticationProvider.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
AuthenticationProvider._create = function(jdel) {var obj = Object.create(AuthenticationProvider.prototype, {});
  AuthenticationProvider.apply(obj, arguments);
  return obj;
}
module.exports = AuthenticationProvider;
var Promise = require('vertx-js/promise');
