/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-auth-common-js/user */
var utils = require('vertx-js/util/utils');
var Authorizations = require('vertx-auth-common-js/authorizations');
var Future = require('vertx-js/future');
var AuthProvider = require('vertx-auth-common-js/auth_provider');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JUser = Java.type('io.vertx.ext.auth.User');

/**
 Represents an authenticates User and contains operations to authorise the user.
 <p>
 Please consult the documentation for a detailed explanation.

 @class
*/
var User = function(j_val) {

  var j_user = j_val;
  var that = this;

  var __super_create = this.create;
  var __super_create = this.create;
  var __super_attributes = this.attributes;
  var __super_expired = this.expired;
  var __super_expired = this.expired;
  var __super_authorizations = this.authorizations;
  var __super_isAuthorized = this.isAuthorized;
  var __super_clearCache = this.clearCache;
  var __super_principal = this.principal;
  var __super_setAuthProvider = this.setAuthProvider;
  /**
   Gets extra attributes of the user. Attributes contains any attributes related
   to the outcome of authenticating a user (e.g.: issued date, metadata, etc...)

   @public

   @return {Object} a json object with any relevant attribute.
   */
  this.attributes =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_user["attributes()"]()) ;
    } else if (typeof __super_attributes != 'undefined') {
      return __super_attributes.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flags this user object to be expired. Expiration takes 3 values in account:

   <ol>
     <li><code>exp</code> "expiration" timestamp in seconds.</li>
     <li><code>iat</code> "issued at" in seconds.</li>
     <li><code>nbf</code> "not before" in seconds.</li>
   </ol>
   A User is considered expired if it contains any of the above and
   the current clock time does not agree with the parameter value. If the {@link User#principal} do not contain a key
   then {@link User#attributes} are checked.
   <p>
   If all of the properties are not available the user will not expire.
   <p>
   Implementations of this interface might relax this rule to account for a leeway to safeguard against
   clock drifting.

   @public
   @param leeway {number} a greater than zero leeway value. 
   @return {boolean} <code>true</code> if expired
   */
  this.expired =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_user["expired()"]() ;
    } else if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_user["expired(int)"](__args[0]) ;
    } else if (typeof __super_expired != 'undefined') {
      return __super_expired.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   returns user's authorizations

   @public

   @return {Authorizations} 
   */
  this.authorizations =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Authorizations, j_user["authorizations()"]()) ;
    } else if (typeof __super_authorizations != 'undefined') {
      return __super_authorizations.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Is the user authorised to

   @public
   @param authority {string} the authority - what this really means is determined by the specific implementation. It might represent a permission to access a resource e.g. `printers:printer34` or it might represent authority to a role in a roles based model, e.g. `role:admin`. 
   @param resultHandler {function} handler that will be called with an {@link AsyncResult} containing the value `true` if the they has the authority or `false` otherwise. 
   @return {User} the User to enable fluent use
   */
  this.isAuthorized =  function(authority, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_user["isAuthorized(java.lang.String,io.vertx.core.Handler)"](authority, function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_user["isAuthorized(java.lang.String,io.vertx.core.Handler)"](authority, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_isAuthorized != 'undefined') {
      return __super_isAuthorized.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The User object will cache any authorities that it knows it has to avoid hitting the
   underlying auth provider each time.  Use this method if you want to clear this cache.

   @public

   @return {User} the User to enable fluent use
   */
  this.clearCache =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_user["clearCache()"]() ;
      return that;
    } else if (typeof __super_clearCache != 'undefined') {
      return __super_clearCache.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the underlying principal for the User. What this actually returns depends on the implementation.
   For a simple user/password based auth, it's likely to contain a JSON object with the following structure:
   <pre>
     {
       "username", "tim"
     }
   </pre>

   @public

   @return {Object} JSON representation of the Principal
   */
  this.principal =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_user["principal()"]()) ;
    } else if (typeof __super_principal != 'undefined') {
      return __super_principal.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the auth provider for the User. This is typically used to reattach a detached User with an AuthProvider, e.g.
   after it has been deserialized.

   @public
   @param authProvider {AuthProvider} the AuthProvider - this must be the same type of AuthProvider that originally created the User 
   */
  this.setAuthProvider =  function(authProvider) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_user["setAuthProvider(io.vertx.ext.auth.AuthProvider)"](authProvider._jdel);
    } else if (typeof __super_setAuthProvider != 'undefined') {
      return __super_setAuthProvider.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_user;
};

User._jclass = utils.getJavaClass("io.vertx.ext.auth.User");
User._jtype = {accept: function(obj) {
    return User._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(User.prototype, {});
    User.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
User._create = function(jdel) {var obj = Object.create(User.prototype, {});
  User.apply(obj, arguments);
  return obj;
}
/**

 @memberof module:vertx-auth-common-js/user
 @param principal {Object} 
 @param attributes {Object} 
 @return {User}
 */
User.create =  function() {
  var __args = arguments;
  if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
    return utils.convReturnVertxGen(User, JUser["create(io.vertx.core.json.JsonObject)"](utils.convParamJsonObject(__args[0]))) ;
  } else if (__args.length === 2 && (typeof __args[0] === 'object' && __args[0] != null) && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(User, JUser["create(io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject)"](utils.convParamJsonObject(__args[0]), utils.convParamJsonObject(__args[1]))) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = User;
var Promise = require('vertx-js/promise');
