/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-auth-oauth2-js/scope_authorization */
var utils = require('vertx-js/util/utils');
var AuthorizationProvider = require('vertx-auth-common-js/authorization_provider');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JScopeAuthorization = Java.type('io.vertx.ext.auth.oauth2.authorization.ScopeAuthorization');

/**
 Scope is a mechanism in OAuth 2.0 to limit an application's access to a user's account.
 An application can request one or more scopes, this information is then presented to the
 user in the consent screen, and the access token issued to the application will be
 limited to the scopes granted.

 The OAuth spec allows the authorization server or user to modify the scopes granted to
 the application compared to what is requested, although there are not many examples of
 services doing this in practice.

 OAuth2 does not define any particular values for scopes, since it is highly dependent
 on the service's internal architecture and needs.

 @class
*/
var ScopeAuthorization = function(j_val) {

  var j_scopeAuthorization = j_val;
  var that = this;
  AuthorizationProvider.call(this, j_val);

  var __super_create = this.create;
  var __super_separator = this.separator;
  var __super_encode = this.encode;
  /**
   Returns the configured separator.

   @public

   @return {string} the separator.
   */
  this.separator =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_scopeAuthorization["separator()"]() ;
    } else if (typeof __super_separator != 'undefined') {
      return __super_separator.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns a String with the given scopes concatenated with the given separator.

   @public
   @param scopes {Array.<string>} a list of scopes 
   @return {string} concatenated string.
   */
  this.encode =  function(scopes) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      return j_scopeAuthorization["encode(java.util.List)"](utils.convParamListBasicOther(scopes)) ;
    } else if (typeof __super_encode != 'undefined') {
      return __super_encode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_scopeAuthorization;
};

ScopeAuthorization._jclass = utils.getJavaClass("io.vertx.ext.auth.oauth2.authorization.ScopeAuthorization");
ScopeAuthorization._jtype = {accept: function(obj) {
    return ScopeAuthorization._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(ScopeAuthorization.prototype, {});
    ScopeAuthorization.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
ScopeAuthorization._create = function(jdel) {var obj = Object.create(ScopeAuthorization.prototype, {});
  ScopeAuthorization.apply(obj, arguments);
  return obj;
}
/**
 Factory method to create a Authorization provider for Oauth 2.0 scopes.

 @memberof module:vertx-auth-oauth2-js/scope_authorization
 @param scopeSeparator {string} the scope separator e.g.: <code>" "</code>, <code>","</code>, <code>"+"</code> 
 @return {ScopeAuthorization} a AuthorizationProvider
 */
ScopeAuthorization.create =  function(scopeSeparator) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'string') {
    return utils.convReturnVertxGen(ScopeAuthorization, JScopeAuthorization["create(java.lang.String)"](scopeSeparator)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = ScopeAuthorization;
var Promise = require('vertx-js/promise');
