/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/json_pointer */
var utils = require('vertx-js/util/utils');
var JsonPointerIterator = require('vertx-js/json_pointer_iterator');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JJsonPointer = Java.type('io.vertx.core.json.pointer.JsonPointer');

/**
 Implementation of <a href="https://tools.ietf.org/html/rfc6901">RFC6901 Json Pointers</a>.

 @class
*/
var JsonPointer = function(j_val) {

  var j_jsonPointer = j_val;
  var that = this;

  var __super_isRootPointer = this.isRootPointer;
  var __super_isLocalPointer = this.isLocalPointer;
  var __super_isParent = this.isParent;
  var __super_toString = this.toString;
  var __super_append = this.append;
  var __super_append = this.append;
  var __super_append = this.append;
  var __super_append = this.append;
  var __super_parent = this.parent;
  var __super_query = this.query;
  var __super_queryOrDefault = this.queryOrDefault;
  var __super_queryJson = this.queryJson;
  var __super_queryJsonOrDefault = this.queryJsonOrDefault;
  var __super_tracedQuery = this.tracedQuery;
  var __super_write = this.write;
  var __super_writeJson = this.writeJson;
  var __super_writeJson = this.writeJson;
  var __super_copy = this.copy;
  var __super_create = this.create;
  var __super_from = this.from;
  /**
   Return <code>true</code> if the pointer is a root pointer

   @public

   @return {boolean}
   */
  this.isRootPointer =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_jsonPointer["isRootPointer()"]() ;
    } else if (typeof __super_isRootPointer != 'undefined') {
      return __super_isRootPointer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return <code>true</code> if the pointer is local (URI with only fragment)

   @public

   @return {boolean}
   */
  this.isLocalPointer =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_jsonPointer["isLocalPointer()"]() ;
    } else if (typeof __super_isLocalPointer != 'undefined') {
      return __super_isLocalPointer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return <code>true</code> if this pointer is a parent pointer of <code>child</code>.
   <br/>
   For instance <code>"/properties"</code> pointer is parent pointer of <code>"/properties/parent"</code>

   @public
   @param child {JsonPointer} 
   @return {boolean}
   */
  this.isParent =  function(child) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      return j_jsonPointer["isParent(io.vertx.core.json.pointer.JsonPointer)"](child._jdel) ;
    } else if (typeof __super_isParent != 'undefined') {
      return __super_isParent.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Build a <a href="https://tools.ietf.org/html/rfc6901#section-5">string representation</a> of the JSON Pointer

   @public

   @return {string}
   */
  this.toString =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_jsonPointer["toString()"]() ;
    } else if (typeof __super_toString != 'undefined') {
      return __super_toString.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Append all tokens of <code>pointer</code> to this pointer <br/>
   Note: The base URI of this pointer will remain untouched

   @public
   @param pointer {JsonPointer} other pointer 
   @return {JsonPointer} a reference to this, so the API can be used fluently
   */
  this.append =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_jsonPointer["append(java.lang.String)"](__args[0]) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] ==='number') {
      j_jsonPointer["append(int)"](__args[0]) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      j_jsonPointer["append(java.util.List)"](utils.convParamListBasicOther(__args[0])) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_jsonPointer["append(io.vertx.core.json.pointer.JsonPointer)"](__args[0]._jdel) ;
      return that;
    } else if (typeof __super_append != 'undefined') {
      return __super_append.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove last reference token of this pointer

   @public

   @return {JsonPointer} a reference to this, so the API can be used fluently
   */
  this.parent =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_jsonPointer["parent()"]() ;
      return that;
    } else if (typeof __super_parent != 'undefined') {
      return __super_parent.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Query <code>objectToQuery</code> using the provided {@link JsonPointerIterator}. <br/>
   If you need to query Vert.x json data structures, use {@link JsonPointer#queryJson}<br/>
   Note: if this pointer is a root pointer, this function returns the provided object

   @public
   @param objectToQuery {Object} the object to query 
   @param iterator {JsonPointerIterator} the json pointer iterator that provides the logic to access to the objectToQuery 
   @return {Object} null if pointer points to not existing value, otherwise the requested value
   */
  this.query =  function(objectToQuery, iterator) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] === 'object' && __args[1]._jdel) {
      return utils.convReturnTypeUnknown(j_jsonPointer["query(java.lang.Object,io.vertx.core.json.pointer.JsonPointerIterator)"](utils.convParamTypeUnknown(objectToQuery), iterator._jdel)) ;
    } else if (typeof __super_query != 'undefined') {
      return __super_query.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Query <code>objectToQuery</code> using the provided {@link JsonPointerIterator}. If the query result is null, returns the default. <br/>
   If you need to query Vert.x json data structures, use {@link JsonPointer#queryJsonOrDefault}<br/>
   Note: if this pointer is a root pointer, this function returns the provided object

   @public
   @param objectToQuery {Object} the object to query 
   @param iterator {JsonPointerIterator} the json pointer iterator that provides the logic to access to the objectToQuery 
   @param defaultValue {Object} default value if query result is null 
   @return {Object} null if pointer points to not existing value, otherwise the requested value
   */
  this.queryOrDefault =  function(objectToQuery, iterator, defaultValue) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] !== 'function' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] !== 'function') {
      return utils.convReturnTypeUnknown(j_jsonPointer["queryOrDefault(java.lang.Object,io.vertx.core.json.pointer.JsonPointerIterator,java.lang.Object)"](utils.convParamTypeUnknown(objectToQuery), iterator._jdel, utils.convParamTypeUnknown(defaultValue))) ;
    } else if (typeof __super_queryOrDefault != 'undefined') {
      return __super_queryOrDefault.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Query <code>jsonElement</code>. <br/>
   Note: if this pointer is a root pointer, this function returns the provided json element

   @public
   @param jsonElement {Object} the json element to query 
   @return {Object} null if pointer points to not existing value, otherwise the requested value
   */
  this.queryJson =  function(jsonElement) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      return utils.convReturnTypeUnknown(j_jsonPointer["queryJson(java.lang.Object)"](utils.convParamTypeUnknown(jsonElement))) ;
    } else if (typeof __super_queryJson != 'undefined') {
      return __super_queryJson.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Query <code>jsonElement</code>. If the query result is null, returns the default.<br/>
   Note: if this pointer is a root pointer, this function returns the provided object

   @public
   @param jsonElement {Object} the json element to query 
   @param defaultValue {Object} default value if query result is null 
   @return {Object} null if pointer points to not existing value, otherwise the requested value
   */
  this.queryJsonOrDefault =  function(jsonElement, defaultValue) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] !== 'function') {
      return utils.convReturnTypeUnknown(j_jsonPointer["queryJsonOrDefault(java.lang.Object,java.lang.Object)"](utils.convParamTypeUnknown(jsonElement), utils.convParamTypeUnknown(defaultValue))) ;
    } else if (typeof __super_queryJsonOrDefault != 'undefined') {
      return __super_queryJsonOrDefault.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Query <code>objectToQuery</code> tracing each element walked during the query, including the first and the result (if any).<br/>
   The first element of the list is objectToQuery and the last is the result, or the element before the first null was encountered

   @public
   @param objectToQuery {Object} the object to query 
   @param iterator {JsonPointerIterator} the json pointer iterator that provides the logic to access to the objectToQuery 
   @return {Array.<Object>} the stream of walked elements
   */
  this.tracedQuery =  function(objectToQuery, iterator) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] === 'object' && __args[1]._jdel) {
      return utils.convReturnListSetObject(j_jsonPointer["tracedQuery(java.lang.Object,io.vertx.core.json.pointer.JsonPointerIterator)"](utils.convParamTypeUnknown(objectToQuery), iterator._jdel)) ;
    } else if (typeof __super_tracedQuery != 'undefined') {
      return __super_tracedQuery.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Write <code>newElement</code> in <code>objectToWrite</code> using this pointer. The path token "-" is handled as append to end of array <br/>
   If you need to write in Vert.x json data structures, use {@link JsonPointer#writeJson} (Object)}<br/>

   @public
   @param objectToWrite {Object} object to write 
   @param iterator {JsonPointerIterator} the json pointer iterator that provides the logic to access to the objectToMutate 
   @param newElement {Object} object to insert 
   @param createOnMissing {boolean} create objects when missing a object key or an array index 
   @return {Object} a reference to objectToWrite if the write was completed, a reference to newElement if the pointer is a root pointer, null if the write failed
   */
  this.write =  function(objectToWrite, iterator, newElement, createOnMissing) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] !== 'function' && typeof __args[1] === 'object' && __args[1]._jdel && typeof __args[2] !== 'function' && typeof __args[3] ==='boolean') {
      return utils.convReturnTypeUnknown(j_jsonPointer["write(java.lang.Object,io.vertx.core.json.pointer.JsonPointerIterator,java.lang.Object,boolean)"](utils.convParamTypeUnknown(objectToWrite), iterator._jdel, utils.convParamTypeUnknown(newElement), createOnMissing)) ;
    } else if (typeof __super_write != 'undefined') {
      return __super_write.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Write <code>newElement</code> in <code>jsonElement</code> using this pointer. The path token "-" is handled as append to end of array.

   @public
   @param jsonElement {Object} json to query and write 
   @param newElement {Object} json to insert 
   @param createOnMissing {boolean} create JsonObject when missing a object key or an array index 
   @return {Object} a reference to json if the write was completed, a reference to newElement if the pointer is a root pointer, null if the write failed
   */
  this.writeJson =  function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] !== 'function') {
      return utils.convReturnTypeUnknown(j_jsonPointer["writeJson(java.lang.Object,java.lang.Object)"](utils.convParamTypeUnknown(__args[0]), utils.convParamTypeUnknown(__args[1]))) ;
    } else if (__args.length === 3 && typeof __args[0] !== 'function' && typeof __args[1] !== 'function' && typeof __args[2] ==='boolean') {
      return utils.convReturnTypeUnknown(j_jsonPointer["writeJson(java.lang.Object,java.lang.Object,boolean)"](utils.convParamTypeUnknown(__args[0]), utils.convParamTypeUnknown(__args[1]), __args[2])) ;
    } else if (typeof __super_writeJson != 'undefined') {
      return __super_writeJson.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Copy a JsonPointer

   @public

   @return {JsonPointer} a copy of this pointer
   */
  this.copy =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(JsonPointer, j_jsonPointer["copy()"]()) ;
    } else if (typeof __super_copy != 'undefined') {
      return __super_copy.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_jsonPointer;
};

JsonPointer._jclass = utils.getJavaClass("io.vertx.core.json.pointer.JsonPointer");
JsonPointer._jtype = {accept: function(obj) {
    return JsonPointer._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(JsonPointer.prototype, {});
    JsonPointer.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
JsonPointer._create = function(jdel) {var obj = Object.create(JsonPointer.prototype, {});
  JsonPointer.apply(obj, arguments);
  return obj;
}
/**
 Build an empty JsonPointer

 @memberof module:vertx-js/json_pointer

 @return {JsonPointer} a new empty JsonPointer
 */
JsonPointer.create =  function() {
  var __args = arguments;
  if (__args.length === 0) {
    return utils.convReturnVertxGen(JsonPointer, JJsonPointer["create()"]()) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

/**
 Build a JsonPointer from a json pointer string

 @memberof module:vertx-js/json_pointer
 @param pointer {string} the string representing a pointer 
 @return {JsonPointer} new instance of JsonPointer
 */
JsonPointer.from =  function(pointer) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'string') {
    return utils.convReturnVertxGen(JsonPointer, JJsonPointer["from(java.lang.String)"](pointer)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = JsonPointer;
var Promise = require('vertx-js/promise');
