/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-js/message */
var utils = require('vertx-js/util/utils');
var MultiMap = require('vertx-js/multi_map');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JMessage = Java.type('io.vertx.core.eventbus.Message');
var DeliveryOptions = Java.type('io.vertx.core.eventbus.DeliveryOptions');

/**
 Represents a message that is received from the event bus in a handler.
 <p>
 @class
*/
var Message = function(j_val, j_arg_0) {

  var j_message = j_val;
  var that = this;
  var j_T = typeof j_arg_0 !== 'undefined' ? j_arg_0 : utils.unknown_jtype;
  var __super_address = this.address;
  var __super_headers = this.headers;
  var __super_body = this.body;
  var __super_replyAddress = this.replyAddress;
  var __super_isSend = this.isSend;
  var __super_reply = this.reply;
  var __super_reply = this.reply;
  var __super_replyAndRequest = this.replyAndRequest;
  var __super_replyAndRequest = this.replyAndRequest;
  var __super_fail = this.fail;
  /**
   The address the message was sent to

   @public

   @return {string}
   */
  this.address =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_message["address()"]() ;
    } else if (typeof __super_address != 'undefined') {
      return __super_address.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Multi-map of message headers. Can be empty

   @public

   @return {MultiMap} the headers
   */
  this.headers =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(MultiMap, j_message["headers()"]()) ;
    } else if (typeof __super_headers != 'undefined') {
      return __super_headers.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The body of the message. Can be null.

   @public

   @return {Object} the body, or null.
   */
  this.body =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedbody == null) {
        that.cachedbody = j_T.wrap(j_message["body()"]());
      }
      return that.cachedbody;
    } else if (typeof __super_body != 'undefined') {
      return __super_body.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The reply address. Can be null.

   @public

   @return {string} the reply address, or null, if message was sent without a reply handler.
   */
  this.replyAddress =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_message["replyAddress()"]() ;
    } else if (typeof __super_replyAddress != 'undefined') {
      return __super_replyAddress.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Signals if this message represents a send or publish event.

   @public

   @return {boolean} true if this is a send.
   */
  this.isSend =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_message["isSend()"]() ;
    } else if (typeof __super_isSend != 'undefined') {
      return __super_isSend.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Link  but allows you to specify delivery options for the reply.

   @public
   @param message {Object} the reply message 
   @param options {Object} the delivery options 
   */
  this.reply =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      j_message["reply(java.lang.Object)"](utils.convParamTypeUnknown(__args[0]));
    } else if (__args.length === 2 && typeof __args[0] !== 'function' && (typeof __args[1] === 'object' && __args[1] != null)) {
      j_message["reply(java.lang.Object,io.vertx.core.eventbus.DeliveryOptions)"](utils.convParamTypeUnknown(__args[0]), __args[1]  != null ? new DeliveryOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null);
    } else if (typeof __super_reply != 'undefined') {
      return __super_reply.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Like  but specifying <code>options</code> that can be used
   to configure the delivery.

   @public
   @param message {Object} the message body, may be <code>null</code> 
   @param options {Object} delivery options 
   @param replyHandler {function} reply handler will be called when any reply from the recipient is received 
   */
  this.replyAndRequest =  function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] !== 'function' && typeof __args[1] === 'function') {
      j_message["replyAndRequest(java.lang.Object,io.vertx.core.Handler)"](utils.convParamTypeUnknown(__args[0]), function(ar) {
        if (ar.succeeded()) {
          __args[1](utils.convReturnVertxGen(Message, ar.result(), undefined), null);
        } else {
          __args[1](null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] !== 'function') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_message["replyAndRequest(java.lang.Object,io.vertx.core.Handler)"](utils.convParamTypeUnknown(__args[0]), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Message, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (__args.length === 3 && typeof __args[0] !== 'function' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_message["replyAndRequest(java.lang.Object,io.vertx.core.eventbus.DeliveryOptions,io.vertx.core.Handler)"](utils.convParamTypeUnknown(__args[0]), __args[1]  != null ? new DeliveryOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null, function(ar) {
        if (ar.succeeded()) {
          __args[2](utils.convReturnVertxGen(Message, ar.result(), undefined), null);
        } else {
          __args[2](null, ar.cause());
        }
      });
    } else if (__args.length === 2 && typeof __args[0] !== 'function' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_message["replyAndRequest(java.lang.Object,io.vertx.core.eventbus.DeliveryOptions,io.vertx.core.Handler)"](utils.convParamTypeUnknown(__args[0]), __args[1]  != null ? new DeliveryOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Message, ar.result(), undefined), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_replyAndRequest != 'undefined') {
      return __super_replyAndRequest.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Signal to the sender that processing of this message failed.
   <p>
   If the message was sent specifying a result handler
   the handler will be called with a failure corresponding to the failure code and message specified here.

   @public
   @param failureCode {number} A failure code to pass back to the sender 
   @param message {string} A message to pass back to the sender 
   */
  this.fail =  function(failureCode, message) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'string') {
      j_message["fail(int,java.lang.String)"](failureCode, message);
    } else if (typeof __super_fail != 'undefined') {
      return __super_fail.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_message;
};

Message._jclass = utils.getJavaClass("io.vertx.core.eventbus.Message");
Message._jtype = {accept: function(obj) {
    return Message._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Message.prototype, {});
    Message.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Message._create = function(jdel) {var obj = Object.create(Message.prototype, {});
  Message.apply(obj, arguments);
  return obj;
}
module.exports = Message;
var Promise = require('vertx-js/promise');
