/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-mongo-js/mongo_client */
var utils = require('vertx-js/util/utils');
var Vertx = require('vertx-js/vertx');
var ReadStream = require('vertx-js/read_stream');
var MongoGridFsClient = require('vertx-mongo-js/mongo_grid_fs_client');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JMongoClient = Java.type('io.vertx.ext.mongo.MongoClient');
var IndexModel = Java.type('io.vertx.ext.mongo.IndexModel');
var MongoClientDeleteResult = Java.type('io.vertx.ext.mongo.MongoClientDeleteResult');
var AggregateOptions = Java.type('io.vertx.ext.mongo.AggregateOptions');
var IndexOptions = Java.type('io.vertx.ext.mongo.IndexOptions');
var BulkOperation = Java.type('io.vertx.ext.mongo.BulkOperation');
var BulkWriteOptions = Java.type('io.vertx.ext.mongo.BulkWriteOptions');
var FindOptions = Java.type('io.vertx.ext.mongo.FindOptions');
var MongoClientBulkWriteResult = Java.type('io.vertx.ext.mongo.MongoClientBulkWriteResult');
var MongoClientUpdateResult = Java.type('io.vertx.ext.mongo.MongoClientUpdateResult');
var UpdateOptions = Java.type('io.vertx.ext.mongo.UpdateOptions');

/**
 A Vert.x service used to interact with MongoDB server instances.
 <p>
 @class
*/
var MongoClient = function(j_val) {

  var j_mongoClient = j_val;
  var that = this;

  var __super_createNonShared = this.createNonShared;
  var __super_createShared = this.createShared;
  var __super_createShared = this.createShared;
  var __super_save = this.save;
  var __super_saveWithOptions = this.saveWithOptions;
  var __super_insert = this.insert;
  var __super_insertWithOptions = this.insertWithOptions;
  var __super_updateCollection = this.updateCollection;
  var __super_updateCollectionWithOptions = this.updateCollectionWithOptions;
  var __super_replaceDocuments = this.replaceDocuments;
  var __super_replaceDocumentsWithOptions = this.replaceDocumentsWithOptions;
  var __super_bulkWrite = this.bulkWrite;
  var __super_bulkWriteWithOptions = this.bulkWriteWithOptions;
  var __super_find = this.find;
  var __super_findBatch = this.findBatch;
  var __super_findWithOptions = this.findWithOptions;
  var __super_findBatchWithOptions = this.findBatchWithOptions;
  var __super_findOne = this.findOne;
  var __super_findOneAndUpdate = this.findOneAndUpdate;
  var __super_findOneAndUpdateWithOptions = this.findOneAndUpdateWithOptions;
  var __super_findOneAndReplace = this.findOneAndReplace;
  var __super_findOneAndReplaceWithOptions = this.findOneAndReplaceWithOptions;
  var __super_findOneAndDelete = this.findOneAndDelete;
  var __super_findOneAndDeleteWithOptions = this.findOneAndDeleteWithOptions;
  var __super_count = this.count;
  var __super_removeDocuments = this.removeDocuments;
  var __super_removeDocumentsWithOptions = this.removeDocumentsWithOptions;
  var __super_removeDocument = this.removeDocument;
  var __super_removeDocumentWithOptions = this.removeDocumentWithOptions;
  var __super_createCollection = this.createCollection;
  var __super_getCollections = this.getCollections;
  var __super_dropCollection = this.dropCollection;
  var __super_createIndex = this.createIndex;
  var __super_createIndexWithOptions = this.createIndexWithOptions;
  var __super_createIndexes = this.createIndexes;
  var __super_listIndexes = this.listIndexes;
  var __super_dropIndex = this.dropIndex;
  var __super_runCommand = this.runCommand;
  var __super_distinct = this.distinct;
  var __super_distinctWithQuery = this.distinctWithQuery;
  var __super_distinctBatch = this.distinctBatch;
  var __super_distinctBatchWithQuery = this.distinctBatchWithQuery;
  var __super_distinctBatchWithQuery = this.distinctBatchWithQuery;
  var __super_aggregate = this.aggregate;
  var __super_aggregateWithOptions = this.aggregateWithOptions;
  var __super_createDefaultGridFsBucketService = this.createDefaultGridFsBucketService;
  var __super_createGridFsBucketService = this.createGridFsBucketService;
  var __super_close = this.close;
  /**
   Save a document in the specified collection
   <p>
   This operation might change <i>_id</i> field of <i>document</i> parameter

   @public
   @param collection {string} the collection 
   @param document {Object} the document 
   @param resultHandler {function} result handler will be provided with the id if document didn't already have one 
   @return {MongoClient}
   */
  this.save =  function(collection, document, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["save(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["save(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_save != 'undefined') {
      return __super_save.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Save a document in the specified collection with the specified write option
   <p>
   This operation might change <i>_id</i> field of <i>document</i> parameter

   @public
   @param collection {string} the collection 
   @param document {Object} the document 
   @param writeOption {Object} the write option to use 
   @param resultHandler {function} result handler will be provided with the id if document didn't already have one 
   @return {MongoClient}
   */
  this.saveWithOptions =  function(collection, document, writeOption, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null) && typeof __args[3] === 'function') {
      j_mongoClient["saveWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["saveWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_saveWithOptions != 'undefined') {
      return __super_saveWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Insert a document in the specified collection
   <p>
   This operation might change <i>_id</i> field of <i>document</i> parameter

   @public
   @param collection {string} the collection 
   @param document {Object} the document 
   @param resultHandler {function} result handler will be provided with the id if document didn't already have one 
   @return {MongoClient}
   */
  this.insert =  function(collection, document, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["insert(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["insert(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_insert != 'undefined') {
      return __super_insert.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Insert a document in the specified collection with the specified write option
   <p>
   This operation might change <i>_id</i> field of <i>document</i> parameter

   @public
   @param collection {string} the collection 
   @param document {Object} the document 
   @param writeOption {Object} the write option to use 
   @param resultHandler {function} result handler will be provided with the id if document didn't already have one 
   @return {MongoClient}
   */
  this.insertWithOptions =  function(collection, document, writeOption, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null) && typeof __args[3] === 'function') {
      j_mongoClient["insertWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["insertWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(document), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_insertWithOptions != 'undefined') {
      return __super_insertWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Update matching documents in the specified collection and return the handler with MongoClientUpdateResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match the documents 
   @param update {Object} used to describe how the documents will be updated 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.updateCollection =  function(collection, query, update, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["updateCollection(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["updateCollection(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_updateCollection != 'undefined') {
      return __super_updateCollection.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Update matching documents in the specified collection, specifying options and return the handler with MongoClientUpdateResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match the documents 
   @param update {Object} used to describe how the documents will be updated 
   @param options {Object} options to configure the update 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.updateCollectionWithOptions =  function(collection, query, update, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && typeof __args[4] === 'function') {
      j_mongoClient["updateCollectionWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), options  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["updateCollectionWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), options  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_updateCollectionWithOptions != 'undefined') {
      return __super_updateCollectionWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Replace matching documents in the specified collection and return the handler with MongoClientUpdateResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match the documents 
   @param replace {Object} all matching documents will be replaced with this 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.replaceDocuments =  function(collection, query, replace, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["replaceDocuments(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["replaceDocuments(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_replaceDocuments != 'undefined') {
      return __super_replaceDocuments.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Replace matching documents in the specified collection, specifying options and return the handler with MongoClientUpdateResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match the documents 
   @param replace {Object} all matching documents will be replaced with this 
   @param options {Object} options to configure the replace 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.replaceDocumentsWithOptions =  function(collection, query, replace, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && typeof __args[4] === 'function') {
      j_mongoClient["replaceDocumentsWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), options  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["replaceDocumentsWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), options  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_replaceDocumentsWithOptions != 'undefined') {
      return __super_replaceDocumentsWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Execute a bulk operation. Can insert, update, replace, and/or delete multiple documents with one request.

   @public
   @param collection {string} the collection 
   @param operations {Array.<Object>} the operations to execute 
   @param resultHandler {function} will be called with a <a href="../../dataobjects.html#MongoClientBulkWriteResult">MongoClientBulkWriteResult</a> when complete 
   @return {MongoClient}
   */
  this.bulkWrite =  function(collection, operations, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && typeof __args[2] === 'function') {
      j_mongoClient["bulkWrite(java.lang.String,java.util.List,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(operations, function(json) { return new BulkOperation(json); }), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["bulkWrite(java.lang.String,java.util.List,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(operations, function(json) { return new BulkOperation(json); }), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_bulkWrite != 'undefined') {
      return __super_bulkWrite.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Execute a bulk operation with the specified write options. Can insert, update, replace, and/or delete multiple
   documents with one request.

   @public
   @param collection {string} the collection 
   @param operations {Array.<Object>} the operations to execute 
   @param bulkWriteOptions {Object} the write options 
   @param resultHandler {function} will be called with a <a href="../../dataobjects.html#MongoClientBulkWriteResult">MongoClientBulkWriteResult</a> when complete 
   @return {MongoClient}
   */
  this.bulkWriteWithOptions =  function(collection, operations, bulkWriteOptions, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["bulkWriteWithOptions(java.lang.String,java.util.List,io.vertx.ext.mongo.BulkWriteOptions,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(operations, function(json) { return new BulkOperation(json); }), bulkWriteOptions  != null ? new BulkWriteOptions(new JsonObject(Java.asJSONCompatible(bulkWriteOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["bulkWriteWithOptions(java.lang.String,java.util.List,io.vertx.ext.mongo.BulkWriteOptions,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(operations, function(json) { return new BulkOperation(json); }), bulkWriteOptions  != null ? new BulkWriteOptions(new JsonObject(Java.asJSONCompatible(bulkWriteOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_bulkWriteWithOptions != 'undefined') {
      return __super_bulkWriteWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find matching documents in the specified collection

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param resultHandler {function} will be provided with list of documents 
   @return {MongoClient}
   */
  this.find =  function(collection, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["find(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["find(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_find != 'undefined') {
      return __super_find.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find matching documents in the specified collection.
   This method use batchCursor for returning each found document.

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @return {ReadStream} a  emitting found documents
   */
  this.findBatch =  function(collection, query) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["findBatch(java.lang.String,io.vertx.core.json.JsonObject)"](collection, utils.convParamJsonObject(query)), undefined) ;
    } else if (typeof __super_findBatch != 'undefined') {
      return __super_findBatch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find matching documents in the specified collection, specifying options

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param options {Object} options to configure the find 
   @param resultHandler {function} will be provided with list of documents 
   @return {MongoClient}
   */
  this.findWithOptions =  function(collection, query, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["findWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), options  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), options  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findWithOptions != 'undefined') {
      return __super_findWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find matching documents in the specified collection, specifying options.
   This method use batchCursor for returning each found document.

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param options {Object} options to configure the find 
   @return {ReadStream} a  emitting found documents
   */
  this.findBatchWithOptions =  function(collection, query, options) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["findBatchWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions)"](collection, utils.convParamJsonObject(query), options  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(options))) : null), undefined) ;
    } else if (typeof __super_findBatchWithOptions != 'undefined') {
      return __super_findBatchWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param fields {Object} the fields 
   @param resultHandler {function} will be provided with the document, if any 
   @return {MongoClient}
   */
  this.findOne =  function(collection, query, fields, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'object' && typeof __args[3] === 'function') {
      j_mongoClient["findOne(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(fields), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'object') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOne(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(fields), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOne != 'undefined') {
      return __super_findOne.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and update it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param update {Object} used to describe how the documents will be updated 
   @param resultHandler {function} will be provided with the document, if any 
   @return {MongoClient}
   */
  this.findOneAndUpdate =  function(collection, query, update, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["findOneAndUpdate(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndUpdate(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndUpdate != 'undefined') {
      return __super_findOneAndUpdate.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and update it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param update {Object} used to describe how the documents will be updated 
   @param findOptions {Object} options to configure the find 
   @param updateOptions {Object} options to configure the update 
   @param resultHandler {function} will be provided with the document, if any 
   @return {MongoClient}
   */
  this.findOneAndUpdateWithOptions =  function(collection, query, update, findOptions, updateOptions, resultHandler) {
    var __args = arguments;
    if (__args.length === 6 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && (typeof __args[4] === 'object' && __args[4] != null) && typeof __args[5] === 'function') {
      j_mongoClient["findOneAndUpdateWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, updateOptions  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(updateOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 5 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && (typeof __args[4] === 'object' && __args[4] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndUpdateWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(update), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, updateOptions  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(updateOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndUpdateWithOptions != 'undefined') {
      return __super_findOneAndUpdateWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and replace it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param replace {Object} the replacement document 
   @param resultHandler {function} will be provided with the document, if any 
   @return {MongoClient}
   */
  this.findOneAndReplace =  function(collection, query, replace, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["findOneAndReplace(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndReplace(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndReplace != 'undefined') {
      return __super_findOneAndReplace.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and replace it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param replace {Object} the replacement document 
   @param findOptions {Object} options to configure the find 
   @param updateOptions {Object} options to configure the update 
   @param resultHandler {function} will be provided with the document, if any 
   @return {MongoClient}
   */
  this.findOneAndReplaceWithOptions =  function(collection, query, replace, findOptions, updateOptions, resultHandler) {
    var __args = arguments;
    if (__args.length === 6 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && (typeof __args[4] === 'object' && __args[4] != null) && typeof __args[5] === 'function') {
      j_mongoClient["findOneAndReplaceWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, updateOptions  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(updateOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 5 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && (typeof __args[3] === 'object' && __args[3] != null) && (typeof __args[4] === 'object' && __args[4] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndReplaceWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.ext.mongo.UpdateOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), utils.convParamJsonObject(replace), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, updateOptions  != null ? new UpdateOptions(new JsonObject(Java.asJSONCompatible(updateOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndReplaceWithOptions != 'undefined') {
      return __super_findOneAndReplaceWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and delete it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param resultHandler {function} will be provided with the deleted document, if any 
   @return {MongoClient}
   */
  this.findOneAndDelete =  function(collection, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["findOneAndDelete(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndDelete(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndDelete != 'undefined') {
      return __super_findOneAndDelete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Find a single matching document in the specified collection and delete it.
   <p>
   This operation might change <i>_id</i> field of <i>query</i> parameter

   @public
   @param collection {string} the collection 
   @param query {Object} the query used to match the document 
   @param findOptions {Object} options to configure the find 
   @param resultHandler {function} will be provided with the deleted document, if any 
   @return {MongoClient}
   */
  this.findOneAndDeleteWithOptions =  function(collection, query, findOptions, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["findOneAndDeleteWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["findOneAndDeleteWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.FindOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), findOptions  != null ? new FindOptions(new JsonObject(Java.asJSONCompatible(findOptions))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_findOneAndDeleteWithOptions != 'undefined') {
      return __super_findOneAndDeleteWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Count matching documents in a collection.

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param resultHandler {function} will be provided with the number of matching documents 
   @return {MongoClient}
   */
  this.count =  function(collection, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["count(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnLong(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["count(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnLong(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_count != 'undefined') {
      return __super_count.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove matching documents from a collection and return the handler with MongoClientDeleteResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.removeDocuments =  function(collection, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["removeDocuments(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["removeDocuments(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_removeDocuments != 'undefined') {
      return __super_removeDocuments.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove matching documents from a collection with the specified write option and return the handler with MongoClientDeleteResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match documents 
   @param writeOption {Object} the write option to use 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.removeDocumentsWithOptions =  function(collection, query, writeOption, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null) && typeof __args[3] === 'function') {
      j_mongoClient["removeDocumentsWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["removeDocumentsWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_removeDocumentsWithOptions != 'undefined') {
      return __super_removeDocumentsWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove a single matching document from a collection and return the handler with MongoClientDeleteResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match document 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.removeDocument =  function(collection, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["removeDocument(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["removeDocument(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_removeDocument != 'undefined') {
      return __super_removeDocument.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove a single matching document from a collection with the specified write option and return the handler with MongoClientDeleteResult result

   @public
   @param collection {string} the collection 
   @param query {Object} query used to match document 
   @param writeOption {Object} the write option to use 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.removeDocumentWithOptions =  function(collection, query, writeOption, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null) && typeof __args[3] === 'function') {
      j_mongoClient["removeDocumentWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'string' || __args[2] == null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["removeDocumentWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.WriteOption,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(query), writeOption == null ? null : io.vertx.ext.mongo.WriteOption.valueOf(writeOption), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnDataObjectAnnotated(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_removeDocumentWithOptions != 'undefined') {
      return __super_removeDocumentWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Create a new collection

   @public
   @param collectionName {string} the name of the collection 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.createCollection =  function(collectionName, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoClient["createCollection(java.lang.String,io.vertx.core.Handler)"](collectionName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createCollection(java.lang.String,io.vertx.core.Handler)"](collectionName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createCollection != 'undefined') {
      return __super_createCollection.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get a list of all collections in the database.

   @public
   @param resultHandler {function} will be called with a list of collections. 
   @return {MongoClient}
   */
  this.getCollections =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mongoClient["getCollections(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          resultHandler(ar.result(), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["getCollections(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(ar.result(), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_getCollections != 'undefined') {
      return __super_getCollections.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Drop a collection

   @public
   @param collection {string} the collection 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.dropCollection =  function(collection, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoClient["dropCollection(java.lang.String,io.vertx.core.Handler)"](collection, function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["dropCollection(java.lang.String,io.vertx.core.Handler)"](collection, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_dropCollection != 'undefined') {
      return __super_dropCollection.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Creates an index.

   @public
   @param collection {string} the collection 
   @param key {Object} A document that contains the field and value pairs where the field is the index key and the value describes the type of index for that field. For an ascending index on a field, specify a value of 1; for descending index, specify a value of -1. 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.createIndex =  function(collection, key, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["createIndex(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(key), function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createIndex(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(key), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createIndex != 'undefined') {
      return __super_createIndex.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Creates an index.

   @public
   @param collection {string} the collection 
   @param key {Object} A document that contains the field and value pairs where the field is the index key and the value describes the type of index for that field. For an ascending index on a field, specify a value of 1; for descending index, specify a value of -1. 
   @param options {Object} the options for the index 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.createIndexWithOptions =  function(collection, key, options, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null) && typeof __args[3] === 'function') {
      j_mongoClient["createIndexWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.IndexOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(key), options  != null ? new IndexOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && (typeof __args[2] === 'object' && __args[2] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createIndexWithOptions(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.ext.mongo.IndexOptions,io.vertx.core.Handler)"](collection, utils.convParamJsonObject(key), options  != null ? new IndexOptions(new JsonObject(Java.asJSONCompatible(options))) : null, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createIndexWithOptions != 'undefined') {
      return __super_createIndexWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   creates an indexes

   @public
   @param collection {string} the collection 
   @param indexes {Array.<Object>} A model that contains pairs of document and indexOptions, document contains the field and value pairs where the field is the index key and the value describes the type of index for that field. For an ascending index on a field, specify a value of 1; for descending index, specify a value of -1. 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.createIndexes =  function(collection, indexes, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && typeof __args[2] === 'function') {
      j_mongoClient["createIndexes(java.lang.String,java.util.List,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(indexes, function(json) { return new IndexModel(json); }), function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createIndexes(java.lang.String,java.util.List,io.vertx.core.Handler)"](collection, utils.convParamListDataObjectAnnotated(indexes, function(json) { return new IndexModel(json); }), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createIndexes != 'undefined') {
      return __super_createIndexes.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get all the indexes in this collection.

   @public
   @param collection {string} the collection 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.listIndexes =  function(collection, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoClient["listIndexes(java.lang.String,io.vertx.core.Handler)"](collection, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["listIndexes(java.lang.String,io.vertx.core.Handler)"](collection, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_listIndexes != 'undefined') {
      return __super_listIndexes.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Drops the index given its name.

   @public
   @param collection {string} the collection 
   @param indexName {string} the name of the index to remove 
   @param resultHandler {function} will be called when complete 
   @return {MongoClient}
   */
  this.dropIndex =  function(collection, indexName, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_mongoClient["dropIndex(java.lang.String,java.lang.String,io.vertx.core.Handler)"](collection, indexName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(null, null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["dropIndex(java.lang.String,java.lang.String,io.vertx.core.Handler)"](collection, indexName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_dropIndex != 'undefined') {
      return __super_dropIndex.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Run an arbitrary MongoDB command.

   @public
   @param commandName {string} the name of the command 
   @param command {Object} the command 
   @param resultHandler {function} will be called with the result. 
   @return {MongoClient}
   */
  this.runCommand =  function(commandName, command, resultHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'function') {
      j_mongoClient["runCommand(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](commandName, utils.convParamJsonObject(command), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && (typeof __args[1] === 'object' && __args[1] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["runCommand(java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](commandName, utils.convParamJsonObject(command), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_runCommand != 'undefined') {
      return __super_runCommand.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the distinct values of the specified field name.
   Return a JsonArray containing distinct values (eg: [ 1 , 89 ])

   @public
   @param collection {string} the collection 
   @param fieldName {string} the field name 
   @param resultClassname {string} 
   @param resultHandler {function} will be provided with array of values. 
   @return {MongoClient}
   */
  this.distinct =  function(collection, fieldName, resultClassname, resultHandler) {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && typeof __args[3] === 'function') {
      j_mongoClient["distinct(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.Handler)"](collection, fieldName, resultClassname, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["distinct(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.Handler)"](collection, fieldName, resultClassname, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_distinct != 'undefined') {
      return __super_distinct.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the distinct values of the specified field name filtered by specified query.
   Return a JsonArray containing distinct values (eg: [ 1 , 89 ])

   @public
   @param collection {string} the collection 
   @param fieldName {string} the field name 
   @param resultClassname {string} 
   @param query {Object} the query 
   @param resultHandler {function} will be provided with array of values. 
   @return {MongoClient}
   */
  this.distinctWithQuery =  function(collection, fieldName, resultClassname, query, resultHandler) {
    var __args = arguments;
    if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && (typeof __args[3] === 'object' && __args[3] != null) && typeof __args[4] === 'function') {
      j_mongoClient["distinctWithQuery(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, fieldName, resultClassname, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnJson(ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && (typeof __args[3] === 'object' && __args[3] != null)) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["distinctWithQuery(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.json.JsonObject,io.vertx.core.Handler)"](collection, fieldName, resultClassname, utils.convParamJsonObject(query), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_distinctWithQuery != 'undefined') {
      return __super_distinctWithQuery.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the distinct values of the specified field name.
   This method use batchCursor for returning each found value.
   Each value is a json fragment with fieldName key (eg: {"num": 1}).

   @public
   @param collection {string} the collection 
   @param fieldName {string} the field name 
   @param resultClassname {string} 
   @return {ReadStream} a  emitting json fragments
   */
  this.distinctBatch =  function(collection, fieldName, resultClassname) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string') {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["distinctBatch(java.lang.String,java.lang.String,java.lang.String)"](collection, fieldName, resultClassname), undefined) ;
    } else if (typeof __super_distinctBatch != 'undefined') {
      return __super_distinctBatch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the distinct values of the specified field name filtered by specified query.
   This method use batchCursor for returning each found value.
   Each value is a json fragment with fieldName key (eg: {"num": 1}).

   @public
   @param collection {string} the collection 
   @param fieldName {string} the field name 
   @param resultClassname {string} 
   @param query {Object} the query 
   @param batchSize {number} the number of documents to load in a batch 
   @return {ReadStream} a  emitting json fragments
   */
  this.distinctBatchWithQuery =  function() {
    var __args = arguments;
    if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && (typeof __args[3] === 'object' && __args[3] != null)) {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["distinctBatchWithQuery(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.json.JsonObject)"](__args[0], __args[1], __args[2], utils.convParamJsonObject(__args[3])), undefined) ;
    } else if (__args.length === 5 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'string' && (typeof __args[3] === 'object' && __args[3] != null) && typeof __args[4] ==='number') {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["distinctBatchWithQuery(java.lang.String,java.lang.String,java.lang.String,io.vertx.core.json.JsonObject,int)"](__args[0], __args[1], __args[2], utils.convParamJsonObject(__args[3]), __args[4]), undefined) ;
    } else if (typeof __super_distinctBatchWithQuery != 'undefined') {
      return __super_distinctBatchWithQuery.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Run aggregate MongoDB command with default <a href="../../dataobjects.html#AggregateOptions">AggregateOptions</a>.

   @public
   @param collection {string} the collection 
   @param pipeline {Array} aggregation pipeline to be executed 
   @return {ReadStream}
   */
  this.aggregate =  function(collection, pipeline) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array) {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["aggregate(java.lang.String,io.vertx.core.json.JsonArray)"](collection, utils.convParamJsonArray(pipeline)), undefined) ;
    } else if (typeof __super_aggregate != 'undefined') {
      return __super_aggregate.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Run aggregate MongoDB command.

   @public
   @param collection {string} the collection 
   @param pipeline {Array} aggregation pipeline to be executed 
   @param options {Object} options to configure the aggregation command 
   @return {ReadStream}
   */
  this.aggregateWithOptions =  function(collection, pipeline, options) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'object' && __args[1] instanceof Array && (typeof __args[2] === 'object' && __args[2] != null)) {
      return utils.convReturnVertxGen(ReadStream, j_mongoClient["aggregateWithOptions(java.lang.String,io.vertx.core.json.JsonArray,io.vertx.ext.mongo.AggregateOptions)"](collection, utils.convParamJsonArray(pipeline), options  != null ? new AggregateOptions(new JsonObject(Java.asJSONCompatible(options))) : null), undefined) ;
    } else if (typeof __super_aggregateWithOptions != 'undefined') {
      return __super_aggregateWithOptions.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Creates a {@link MongoGridFsClient} used to interact with Mongo GridFS.

   @public
   @param resultHandler {function} the {@link MongoGridFsClient} to interact with the bucket named bucketName 
   @return {MongoClient}
   */
  this.createDefaultGridFsBucketService =  function(resultHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mongoClient["createDefaultGridFsBucketService(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnVertxGen(MongoGridFsClient, ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createDefaultGridFsBucketService(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(MongoGridFsClient, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createDefaultGridFsBucketService != 'undefined') {
      return __super_createDefaultGridFsBucketService.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Creates a {@link MongoGridFsClient} used to interact with Mongo GridFS.

   @public
   @param bucketName {string} the name of the GridFS bucket 
   @param resultHandler {function} the {@link MongoGridFsClient} to interact with the bucket named bucketName 
   @return {MongoClient}
   */
  this.createGridFsBucketService =  function(bucketName, resultHandler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_mongoClient["createGridFsBucketService(java.lang.String,io.vertx.core.Handler)"](bucketName, function(ar) {
        if (ar.succeeded()) {
          resultHandler(utils.convReturnVertxGen(MongoGridFsClient, ar.result()), null);
        } else {
          resultHandler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["createGridFsBucketService(java.lang.String,io.vertx.core.Handler)"](bucketName, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(MongoGridFsClient, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_createGridFsBucketService != 'undefined') {
      return __super_createGridFsBucketService.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Close the client and release its resources

   @public
   @param handler {function} 
   */
  this.close =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_mongoClient["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_mongoClient["close(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_mongoClient;
};

MongoClient._jclass = utils.getJavaClass("io.vertx.ext.mongo.MongoClient");
MongoClient._jtype = {accept: function(obj) {
    return MongoClient._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(MongoClient.prototype, {});
    MongoClient.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
MongoClient._create = function(jdel) {var obj = Object.create(MongoClient.prototype, {});
  MongoClient.apply(obj, arguments);
  return obj;
}
/**
 Create a Mongo client which maintains its own data source.

 @memberof module:vertx-mongo-js/mongo_client
 @param vertx {Vertx} the Vert.x instance 
 @param config {Object} the configuration 
 @return {MongoClient} the client
 */
MongoClient.createNonShared =  function(vertx, config) {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(MongoClient, JMongoClient["createNonShared(io.vertx.core.Vertx,io.vertx.core.json.JsonObject)"](vertx._jdel, utils.convParamJsonObject(config))) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

/**
 Like {@link MongoClient#createShared} but with the default data source name

 @memberof module:vertx-mongo-js/mongo_client
 @param vertx {Vertx} the Vert.x instance 
 @param config {Object} the configuration 
 @return {MongoClient} the client
 */
MongoClient.createShared =  function() {
  var __args = arguments;
  if (__args.length === 3 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null) && typeof __args[2] === 'string') {
    return utils.convReturnVertxGen(MongoClient, JMongoClient["createShared(io.vertx.core.Vertx,io.vertx.core.json.JsonObject,java.lang.String)"](__args[0]._jdel, utils.convParamJsonObject(__args[1]), __args[2])) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(MongoClient, JMongoClient["createShared(io.vertx.core.Vertx,io.vertx.core.json.JsonObject)"](__args[0]._jdel, utils.convParamJsonObject(__args[1]))) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

MongoClient.DEFAULT_POOL_NAME = JMongoClient.DEFAULT_POOL_NAME;
MongoClient.DEFAULT_DB_NAME = JMongoClient.DEFAULT_DB_NAME;
module.exports = MongoClient;
var Promise = require('vertx-js/promise');
