/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-redis-js/redis */
var utils = require('vertx-js/util/utils');
var Response = require('vertx-redis-js/response');
var Vertx = require('vertx-js/vertx');
var RedisConnection = require('vertx-redis-js/redis_connection');
var Future = require('vertx-js/future');
var Request = require('vertx-redis-js/request');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRedis = Java.type('io.vertx.redis.client.Redis');
var RedisOptions = Java.type('io.vertx.redis.client.RedisOptions');

/**
 A simple Redis client.

 @class
*/
var Redis = function(j_val) {

  var j_redis = j_val;
  var that = this;

  var __super_createClient = this.createClient;
  var __super_createClient = this.createClient;
  var __super_createClient = this.createClient;
  var __super_connect = this.connect;
  var __super_close = this.close;
  var __super_send = this.send;
  var __super_batch = this.batch;
  /**
   Connects to the redis server.

   @public
   @param handler {function} the async result handler 
   @return {Redis} a reference to this, so the API can be used fluently
   */
  this.connect =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_redis["connect(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnVertxGen(RedisConnection, ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 0) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["connect(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(RedisConnection, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_connect != 'undefined') {
      return __super_connect.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Closes the client and terminates any connection.

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redis["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Send the given command to the redis server or cluster.

   @public
   @param command {Request} the command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {Redis} fluent self.
   */
  this.send =  function(command, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_redis["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_send != 'undefined') {
      return __super_send.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends a list of commands in a single IO operation, this prevents any inter twinning to happen from other
   client users.

   @public
   @param commands {Array.<Request>} list of command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {Redis} fluent self.
   */
  this.batch =  function(commands, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_redis["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redis["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_batch != 'undefined') {
      return __super_batch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_redis;
};

Redis._jclass = utils.getJavaClass("io.vertx.redis.client.Redis");
Redis._jtype = {accept: function(obj) {
    return Redis._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Redis.prototype, {});
    Redis.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Redis._create = function(jdel) {var obj = Object.create(Redis.prototype, {});
  Redis.apply(obj, arguments);
  return obj;
}
/**
 Create a new redis client using the given client options.

 @memberof module:vertx-redis-js/redis
 @param vertx {Vertx} the vertx instance 
 @param options {Object} the user provided options 
 @return {Redis} the client
 */
Redis.createClient =  function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(Redis, JRedis["createClient(io.vertx.core.Vertx)"](__args[0]._jdel)) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
    return utils.convReturnVertxGen(Redis, JRedis["createClient(io.vertx.core.Vertx,java.lang.String)"](__args[0]._jdel, __args[1])) ;
  } else if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(Redis, JRedis["createClient(io.vertx.core.Vertx,io.vertx.redis.client.RedisOptions)"](__args[0]._jdel, __args[1]  != null ? new RedisOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = Redis;
var Promise = require('vertx-js/promise');
