/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-redis-js/redis_connection */
var utils = require('vertx-js/util/utils');
var Pipe = require('vertx-js/pipe');
var Response = require('vertx-redis-js/response');
var WriteStream = require('vertx-js/write_stream');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');
var Request = require('vertx-redis-js/request');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRedisConnection = Java.type('io.vertx.redis.client.RedisConnection');

/**
 A simple Redis client.

 @class
*/
var RedisConnection = function(j_val) {

  var j_redisConnection = j_val;
  var that = this;
  ReadStream.call(this, j_val);

  var __super_pipe = this.pipe;
  var __super_pipeTo = this.pipeTo;
  var __super_exceptionHandler = this.exceptionHandler;
  var __super_handler = this.handler;
  var __super_pause = this.pause;
  var __super_resume = this.resume;
  var __super_fetch = this.fetch;
  var __super_endHandler = this.endHandler;
  var __super_send = this.send;
  var __super_batch = this.batch;
  var __super_close = this.close;
  var __super_pendingQueueFull = this.pendingQueueFull;
  /**
   Pause this stream and return a  to transfer the elements of this stream to a destination .
   <p/>
   The stream will be resumed when the pipe will be wired to a <code>WriteStream</code>.

   @public

   @return {Pipe} a pipe
   */
  this.pipe =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Pipe, j_redisConnection["pipe()"](), Response._jtype) ;
    } else if (typeof __super_pipe != 'undefined') {
      return __super_pipe.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Pipe this <code>ReadStream</code> to the <code>WriteStream</code>.
   <p>
   Elements emitted by this stream will be written to the write stream until this stream ends or fails.
   <p>
   Once this stream has ended or failed, the write stream will be ended and the <code>handler</code> will be
   called with the result.

   @public
   @param dst {WriteStream} the destination write stream 
   @param handler {function} 
   */
  this.pipeTo =  function(dst, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_redisConnection["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redisConnection["pipeTo(io.vertx.core.streams.WriteStream,io.vertx.core.Handler)"](dst._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_pipeTo != 'undefined') {
      return __super_pipeTo.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public
   @param handler {function} 
   @return {RedisConnection}
   */
  this.exceptionHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redisConnection["exceptionHandler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnThrowable(jVal));
      }) ;
      return that;
    } else if (typeof __super_exceptionHandler != 'undefined') {
      return __super_exceptionHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public
   @param handler {function} 
   @return {RedisConnection}
   */
  this.handler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redisConnection["handler(io.vertx.core.Handler)"](handler == null ? null : function(jVal) {
        handler(utils.convReturnVertxGen(Response, jVal));
      }) ;
      return that;
    } else if (typeof __super_handler != 'undefined') {
      return __super_handler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public

   @return {RedisConnection}
   */
  this.pause =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redisConnection["pause()"]() ;
      return that;
    } else if (typeof __super_pause != 'undefined') {
      return __super_pause.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public

   @return {RedisConnection}
   */
  this.resume =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redisConnection["resume()"]() ;
      return that;
    } else if (typeof __super_resume != 'undefined') {
      return __super_resume.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public
   @param amount {number} 
   @return {RedisConnection}
   */
  this.fetch =  function(amount) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_redisConnection["fetch(long)"](amount) ;
      return that;
    } else if (typeof __super_fetch != 'undefined') {
      return __super_fetch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   

   @public
   @param endHandler {function} 
   @return {RedisConnection}
   */
  this.endHandler =  function(endHandler) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'function' || __args[0] == null)) {
      j_redisConnection["endHandler(io.vertx.core.Handler)"](endHandler) ;
      return that;
    } else if (typeof __super_endHandler != 'undefined') {
      return __super_endHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Send the given command to the redis server or cluster.

   @public
   @param command {Request} the command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {RedisConnection} fluent self.
   */
  this.send =  function(command, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_redisConnection["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redisConnection["send(io.vertx.redis.client.Request,io.vertx.core.Handler)"](command._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnVertxGen(Response, ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_send != 'undefined') {
      return __super_send.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Sends a list of commands in a single IO operation, this prevents any inter twinning to happen from other
   client users.

   @public
   @param commands {Array.<Request>} list of command to send 
   @param onSend {function} the asynchronous result handler. 
   @return {RedisConnection} fluent self.
   */
  this.batch =  function(commands, onSend) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'function') {
      j_redisConnection["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          onSend(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          onSend(null, ar.cause());
        }
      }) ;
      return that;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_redisConnection["batch(java.util.List,io.vertx.core.Handler)"](utils.convParamListVertxGen(commands), function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnListSetVertxGen(ar.result(), Response), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_batch != 'undefined') {
      return __super_batch.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Closes the connection or returns to the pool.

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_redisConnection["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Flag to notify if the pending message queue (commands in transit) is full.

   When the pending message queue is full and a new send command is issued it will result in a exception to be thrown.
   Checking this flag before sending can allow the application to wait before sending the next message.

   @public

   @return {boolean} true is queue is full.
   */
  this.pendingQueueFull =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_redisConnection["pendingQueueFull()"]() ;
    } else if (typeof __super_pendingQueueFull != 'undefined') {
      return __super_pendingQueueFull.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_redisConnection;
};

RedisConnection._jclass = utils.getJavaClass("io.vertx.redis.client.RedisConnection");
RedisConnection._jtype = {accept: function(obj) {
    return RedisConnection._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(RedisConnection.prototype, {});
    RedisConnection.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
RedisConnection._create = function(jdel) {var obj = Object.create(RedisConnection.prototype, {});
  RedisConnection.apply(obj, arguments);
  return obj;
}
module.exports = RedisConnection;
var Promise = require('vertx-js/promise');
