/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-js/o_auth2_auth_handler */
var utils = require('vertx-js/util/utils');
var Route = require('vertx-web-js/route');
var AuthHandler = require('vertx-web-js/auth_handler');
var User = require('vertx-auth-common-js/user');
var RoutingContext = require('vertx-web-js/routing_context');
var OAuth2Auth = require('vertx-auth-oauth2-js/o_auth2_auth');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JOAuth2AuthHandler = Java.type('io.vertx.ext.web.handler.OAuth2AuthHandler');

/**
 An auth handler that provides OAuth2 Authentication support. This handler is suitable for AuthCode flows.

 @class
*/
var OAuth2AuthHandler = function(j_val) {

  var j_oAuth2AuthHandler = j_val;
  var that = this;
  AuthHandler.call(this, j_val);

  var __super_handle = this.handle;
  var __super_parseCredentials = this.parseCredentials;
  var __super_addAuthority = this.addAuthority;
  var __super_addAuthorities = this.addAuthorities;
  var __super_authorize = this.authorize;
  var __super_create = this.create;
  var __super_create = this.create;
  var __super_extraParams = this.extraParams;
  var __super_setupCallback = this.setupCallback;
  /**
   Something has happened, so handle it.

   @public
   @param event {RoutingContext} the event to handle 
   */
  this.handle =  function(event) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_oAuth2AuthHandler["handle(io.vertx.ext.web.RoutingContext)"](event._jdel);
    } else if (typeof __super_handle != 'undefined') {
      return __super_handle.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Parses the credentials from the request into a JsonObject. The implementation should
   be able to extract the required info for the auth provider in the format the provider
   expects.

   @public
   @param context {RoutingContext} the routing context 
   @param handler {function} the handler to be called once the information is available. 
   */
  this.parseCredentials =  function(context, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_oAuth2AuthHandler["parseCredentials(io.vertx.ext.web.RoutingContext,io.vertx.core.Handler)"](context._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(utils.convReturnJson(ar.result()), null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_oAuth2AuthHandler["parseCredentials(io.vertx.ext.web.RoutingContext,io.vertx.core.Handler)"](context._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(utils.convReturnJson(ar.result()), null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_parseCredentials != 'undefined') {
      return __super_parseCredentials.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a required authority for this auth handler

   @public
   @param authority {string} the authority 
   @return {AuthHandler} a reference to this, so the API can be used fluently
   */
  this.addAuthority =  function(authority) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_oAuth2AuthHandler["addAuthority(java.lang.String)"](authority) ;
      return that;
    } else if (typeof __super_addAuthority != 'undefined') {
      return __super_addAuthority.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a set of required authorities for this auth handler

   @public
   @param authorities {Array.<string>} the set of authorities 
   @return {AuthHandler} a reference to this, so the API can be used fluently
   */
  this.addAuthorities =  function(authorities) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      j_oAuth2AuthHandler["addAuthorities(java.util.Set)"](utils.convParamSetBasicOther(authorities)) ;
      return that;
    } else if (typeof __super_addAuthorities != 'undefined') {
      return __super_addAuthorities.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Authorizes the given user against all added authorities.

   @public
   @param user {User} a user. 
   @param handler {function} the handler for the result. 
   */
  this.authorize =  function(user, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'function') {
      j_oAuth2AuthHandler["authorize(io.vertx.ext.auth.User,io.vertx.core.Handler)"](user._jdel, function(ar) {
        if (ar.succeeded()) {
          handler(null, null);
        } else {
          handler(null, ar.cause());
        }
      });
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      var __prom = Promise.promise();
      var __prom_completer_handler = function (result, cause) { if (cause === null) { __prom.complete(result); } else { __prom.fail(cause); } };
      j_oAuth2AuthHandler["authorize(io.vertx.ext.auth.User,io.vertx.core.Handler)"](user._jdel, function(ar) {
        if (ar.succeeded()) {
          __prom_completer_handler(null, null);
        } else {
          __prom_completer_handler(null, ar.cause());
        }
      });
      return __prom.future();
    } else if (typeof __super_authorize != 'undefined') {
      return __super_authorize.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Extra parameters needed to be passed while requesting a token.

   @public
   @param extraParams {Object} extra optional parameters. 
   @return {OAuth2AuthHandler} self
   */
  this.extraParams =  function(extraParams) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      j_oAuth2AuthHandler["extraParams(io.vertx.core.json.JsonObject)"](utils.convParamJsonObject(extraParams)) ;
      return that;
    } else if (typeof __super_extraParams != 'undefined') {
      return __super_extraParams.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   add the callback handler to a given route.

   @public
   @param route {Route} a given route e.g.: `/callback` 
   @return {OAuth2AuthHandler} self
   */
  this.setupCallback =  function(route) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_oAuth2AuthHandler["setupCallback(io.vertx.ext.web.Route)"](route._jdel) ;
      return that;
    } else if (typeof __super_setupCallback != 'undefined') {
      return __super_setupCallback.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_oAuth2AuthHandler;
};

OAuth2AuthHandler._jclass = utils.getJavaClass("io.vertx.ext.web.handler.OAuth2AuthHandler");
OAuth2AuthHandler._jtype = {accept: function(obj) {
    return OAuth2AuthHandler._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(OAuth2AuthHandler.prototype, {});
    OAuth2AuthHandler.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
OAuth2AuthHandler._create = function(jdel) {var obj = Object.create(OAuth2AuthHandler.prototype, {});
  OAuth2AuthHandler.apply(obj, arguments);
  return obj;
}
/**
 Create a OAuth2 auth handler without host pinning.
 Most providers will not look to the redirect url but always redirect to
 the preconfigured callback. So this factory does not provide a callback url.

 @memberof module:vertx-web-js/o_auth2_auth_handler
 @param authProvider {OAuth2Auth} the auth provider to use 
 @return {OAuth2AuthHandler} the auth handler
 */
OAuth2AuthHandler.create =  function() {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string') {
    return utils.convReturnVertxGen(OAuth2AuthHandler, JOAuth2AuthHandler["create(io.vertx.ext.auth.oauth2.OAuth2Auth,java.lang.String)"](__args[0]._jdel, __args[1])) ;
  } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(OAuth2AuthHandler, JOAuth2AuthHandler["create(io.vertx.ext.auth.oauth2.OAuth2Auth)"](__args[0]._jdel)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = OAuth2AuthHandler;
var Promise = require('vertx-js/promise');
