/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-js/routing_context */
var utils = require('vertx-js/util/utils');
var Route = require('vertx-web-js/route');
var HttpServerRequest = require('vertx-js/http_server_request');
var Session = require('vertx-web-js/session');
var MultiMap = require('vertx-js/multi_map');
var Vertx = require('vertx-js/vertx');
var Future = require('vertx-js/future');
var LanguageHeader = require('vertx-web-js/language_header');
var FileUpload = require('vertx-web-js/file_upload');
var User = require('vertx-auth-common-js/user');
var Buffer = require('vertx-js/buffer');
var HttpServerResponse = require('vertx-js/http_server_response');
var ParsedHeaderValues = require('vertx-web-js/parsed_header_values');
var Cookie = require('vertx-js/cookie');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JsonArray = io.vertx.core.json.JsonArray;
var JRoutingContext = Java.type('io.vertx.ext.web.RoutingContext');

/**
 Represents the context for the handling of a request in Vert.x-Web.
 <p>
 A new instance is created for each HTTP request that is received in the
 @class
*/
var RoutingContext = function(j_val) {

  var j_routingContext = j_val;
  var that = this;

  var __super_request = this.request;
  var __super_response = this.response;
  var __super_next = this.next;
  var __super_fail = this.fail;
  var __super_fail = this.fail;
  var __super_fail = this.fail;
  var __super_put = this.put;
  var __super_get = this.get;
  var __super_remove = this.remove;
  var __super_vertx = this.vertx;
  var __super_mountPoint = this.mountPoint;
  var __super_currentRoute = this.currentRoute;
  var __super_normalisedPath = this.normalisedPath;
  var __super_getCookie = this.getCookie;
  var __super_addCookie = this.addCookie;
  var __super_removeCookie = this.removeCookie;
  var __super_removeCookie = this.removeCookie;
  var __super_cookieCount = this.cookieCount;
  var __super_cookieMap = this.cookieMap;
  var __super_getBodyAsString = this.getBodyAsString;
  var __super_getBodyAsString = this.getBodyAsString;
  var __super_getBodyAsJson = this.getBodyAsJson;
  var __super_getBodyAsJsonArray = this.getBodyAsJsonArray;
  var __super_getBody = this.getBody;
  var __super_fileUploads = this.fileUploads;
  var __super_session = this.session;
  var __super_user = this.user;
  var __super_failure = this.failure;
  var __super_statusCode = this.statusCode;
  var __super_getAcceptableContentType = this.getAcceptableContentType;
  var __super_parsedHeaders = this.parsedHeaders;
  var __super_addHeadersEndHandler = this.addHeadersEndHandler;
  var __super_removeHeadersEndHandler = this.removeHeadersEndHandler;
  var __super_addBodyEndHandler = this.addBodyEndHandler;
  var __super_removeBodyEndHandler = this.removeBodyEndHandler;
  var __super_failed = this.failed;
  var __super_setBody = this.setBody;
  var __super_setSession = this.setSession;
  var __super_setUser = this.setUser;
  var __super_clearUser = this.clearUser;
  var __super_setAcceptableContentType = this.setAcceptableContentType;
  var __super_reroute = this.reroute;
  var __super_reroute = this.reroute;
  var __super_acceptableLanguages = this.acceptableLanguages;
  var __super_preferredLanguage = this.preferredLanguage;
  var __super_pathParams = this.pathParams;
  var __super_pathParam = this.pathParam;
  var __super_queryParams = this.queryParams;
  var __super_queryParam = this.queryParam;
  var __super_attachment = this.attachment;
  var __super_redirect = this.redirect;
  var __super_json = this.json;
  var __super_is = this.is;
  var __super_isFresh = this.isFresh;
  var __super_etag = this.etag;
  var __super_lastModified = this.lastModified;
  var __super_end = this.end;
  var __super_end = this.end;
  var __super_end = this.end;
  /**

   @public

   @return {HttpServerRequest} the HTTP request object
   */
  this.request =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedrequest == null) {
        that.cachedrequest = utils.convReturnVertxGen(HttpServerRequest, j_routingContext["request()"]());
      }
      return that.cachedrequest;
    } else if (typeof __super_request != 'undefined') {
      return __super_request.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {HttpServerResponse} the HTTP response object
   */
  this.response =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedresponse == null) {
        that.cachedresponse = utils.convReturnVertxGen(HttpServerResponse, j_routingContext["response()"]());
      }
      return that.cachedresponse;
    } else if (typeof __super_response != 'undefined') {
      return __super_response.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Tell the router to route this context to the next matching route (if any).
   This method, if called, does not need to be called during the execution of the handler, it can be called
   some arbitrary time later, if required.
   <p>
   If next is not called for a handler then the handler should make sure it ends the response or no response
   will be sent.

   @public

   */
  this.next =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_routingContext["next()"]();
    } else if (typeof __super_next != 'undefined') {
      return __super_next.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Fail the context with the specified throwable and the specified the status code.
   <p>
   This will cause the router to route the context to any matching failure handlers for the request. If no failure handlers
   match It will trigger the error handler matching the status code. You can define such error handler with
   {@link Router#errorHandler}. If no error handler is not defined, It will send a default failure response with provided status code.

   @public
   @param statusCode {number} the HTTP status code 
   @param throwable {todo} a throwable representing the failure 
   */
  this.fail =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      j_routingContext["fail(int)"](__args[0]);
    } else if (__args.length === 1 && typeof __args[0] === 'object') {
      j_routingContext["fail(java.lang.Throwable)"](utils.convParamThrowable(__args[0]));
    } else if (__args.length === 2 && typeof __args[0] ==='number' && typeof __args[1] === 'object') {
      j_routingContext["fail(int,java.lang.Throwable)"](__args[0], utils.convParamThrowable(__args[1]));
    } else if (typeof __super_fail != 'undefined') {
      return __super_fail.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Put some arbitrary data in the context. This will be available in any handlers that receive the context.

   @public
   @param key {string} the key for the data 
   @param obj {Object} the data 
   @return {RoutingContext} a reference to this, so the API can be used fluently
   */
  this.put =  function(key, obj) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] !== 'function') {
      j_routingContext["put(java.lang.String,java.lang.Object)"](key, utils.convParamTypeUnknown(obj)) ;
      return that;
    } else if (typeof __super_put != 'undefined') {
      return __super_put.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get some data from the context. The data is available in any handlers that receive the context.

   @public
   @param key {string} the key for the data 
   @return {Object} the data
   */
  this.get =  function(key) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnTypeUnknown(j_routingContext["get(java.lang.String)"](key)) ;
    } else if (typeof __super_get != 'undefined') {
      return __super_get.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove some data from the context. The data is available in any handlers that receive the context.

   @public
   @param key {string} the key for the data 
   @return {Object} the previous data associated with the key
   */
  this.remove =  function(key) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnTypeUnknown(j_routingContext["remove(java.lang.String)"](key)) ;
    } else if (typeof __super_remove != 'undefined') {
      return __super_remove.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Vertx} the Vert.x instance associated to the initiating {@link Router} for this context
   */
  this.vertx =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedvertx == null) {
        that.cachedvertx = utils.convReturnVertxGen(Vertx, j_routingContext["vertx()"]());
      }
      return that.cachedvertx;
    } else if (typeof __super_vertx != 'undefined') {
      return __super_vertx.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {string} the mount point for this router. It will be null for a top level router. For a sub-router it will be the path at which the subrouter was mounted.
   */
  this.mountPoint =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["mountPoint()"]() ;
    } else if (typeof __super_mountPoint != 'undefined') {
      return __super_mountPoint.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Route} the current route this context is being routed through.
   */
  this.currentRoute =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Route, j_routingContext["currentRoute()"]()) ;
    } else if (typeof __super_currentRoute != 'undefined') {
      return __super_currentRoute.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return the normalised path for the request.
   <p>
   The normalised path is where the URI path has been decoded, i.e. any unicode or other illegal URL characters that
   were encoded in the original URL with `%` will be returned to their original form. E.g. `%20` will revert to a space.
   Also `+` reverts to a space in a query.
   <p>
   The normalised path will also not contain any `..` character sequences to prevent resources being accessed outside
   of the permitted area.
   <p>
   It's recommended to always use the normalised path as opposed to 
   if accessing server resources requested by a client.

   @public

   @return {string} the normalised path
   */
  this.normalisedPath =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["normalisedPath()"]() ;
    } else if (typeof __super_normalisedPath != 'undefined') {
      return __super_normalisedPath.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the cookie with the specified name.

   @public
   @param name {string} the cookie name 
   @return {Cookie} the cookie
   */
  this.getCookie =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Cookie, j_routingContext["getCookie(java.lang.String)"](name)) ;
    } else if (typeof __super_getCookie != 'undefined') {
      return __super_getCookie.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a cookie. This will be sent back to the client in the response.

   @public
   @param cookie {Cookie} the cookie 
   @return {RoutingContext} a reference to this, so the API can be used fluently
   */
  this.addCookie =  function(cookie) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_routingContext["addCookie(io.vertx.core.http.Cookie)"](cookie._jdel) ;
      return that;
    } else if (typeof __super_addCookie != 'undefined') {
      return __super_addCookie.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove a cookie from the cookie set. If invalidate is true then it will expire a cookie, notifying a User Agent to
   remove it from its cookie jar.

   @public
   @param name {string} the name of the cookie 
   @param invalidate {boolean} 
   @return {Cookie} the cookie, if it existed, or null
   */
  this.removeCookie =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Cookie, j_routingContext["removeCookie(java.lang.String)"](__args[0])) ;
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] ==='boolean') {
      return utils.convReturnVertxGen(Cookie, j_routingContext["removeCookie(java.lang.String,boolean)"](__args[0], __args[1])) ;
    } else if (typeof __super_removeCookie != 'undefined') {
      return __super_removeCookie.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {number} the number of cookies.
   */
  this.cookieCount =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["cookieCount()"]() ;
    } else if (typeof __super_cookieCount != 'undefined') {
      return __super_cookieCount.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Object.<string, Cookie>} a map of all the cookies.
   */
  this.cookieMap =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMapVertxGen(j_routingContext["cookieMap()"](), Cookie) ;
    } else if (typeof __super_cookieMap != 'undefined') {
      return __super_cookieMap.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the entire HTTP request body as a string, assuming the specified encoding. The context must have first been routed to a
   {@link BodyHandler} for this to be populated.

   @public
   @param encoding {string} the encoding, e.g. "UTF-16" 
   @return {string} the body
   */
  this.getBodyAsString =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["getBodyAsString()"]() ;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_routingContext["getBodyAsString(java.lang.String)"](__args[0]) ;
    } else if (typeof __super_getBodyAsString != 'undefined') {
      return __super_getBodyAsString.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Object} Get the entire HTTP request body as a . The context must have first been routed to a {@link BodyHandler} for this to be populated. <br/> When the body is <code>null</code> or the <code>"null"</code> JSON literal then <code>null</code> is returned.
   */
  this.getBodyAsJson =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_routingContext["getBodyAsJson()"]()) ;
    } else if (typeof __super_getBodyAsJson != 'undefined') {
      return __super_getBodyAsJson.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Array} Get the entire HTTP request body as a . The context must have first been routed to a {@link BodyHandler} for this to be populated. <br/> When the body is <code>null</code> or the <code>"null"</code> JSON literal then <code>null</code> is returned.
   */
  this.getBodyAsJsonArray =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_routingContext["getBodyAsJsonArray()"]()) ;
    } else if (typeof __super_getBodyAsJsonArray != 'undefined') {
      return __super_getBodyAsJsonArray.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Buffer} Get the entire HTTP request body as a . The context must have first been routed to a {@link BodyHandler} for this to be populated.
   */
  this.getBody =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Buffer, j_routingContext["getBody()"]()) ;
    } else if (typeof __super_getBody != 'undefined') {
      return __super_getBody.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Array.<FileUpload>} a set of fileuploads (if any) for the request. The context must have first been routed to a {@link BodyHandler} for this to work.
   */
  this.fileUploads =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnListSetVertxGen(j_routingContext["fileUploads()"](), FileUpload) ;
    } else if (typeof __super_fileUploads != 'undefined') {
      return __super_fileUploads.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the session. The context must have first been routed to a {@link SessionHandler}
   for this to be populated.
   Sessions live for a browser session, and are maintained by session cookies.

   @public

   @return {Session} the session.
   */
  this.session =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Session, j_routingContext["session()"]()) ;
    } else if (typeof __super_session != 'undefined') {
      return __super_session.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the authenticated user (if any). This will usually be injected by an auth handler if authentication if successful.

   @public

   @return {User} the user, or null if the current user is not authenticated.
   */
  this.user =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(User, j_routingContext["user()"]()) ;
    } else if (typeof __super_user != 'undefined') {
      return __super_user.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   If the context is being routed to failure handlers after a failure has been triggered by calling
   {@link RoutingContext#fail} then this will return that throwable. It can be used by failure handlers to render a response,
   e.g. create a failure response page.

   @public

   @return {todo} the throwable used when signalling failure
   */
  this.failure =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedfailure == null) {
        that.cachedfailure = utils.convReturnThrowable(j_routingContext["failure()"]());
      }
      return that.cachedfailure;
    } else if (typeof __super_failure != 'undefined') {
      return __super_failure.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   If the context is being routed to failure handlers after a failure has been triggered by calling
   {@link RoutingContext#fail}  then this will return that status code.  It can be used by failure handlers to render a response,
   e.g. create a failure response page.

   When the status code has not been set yet (it is undefined) its value will be -1.

   @public

   @return {number} the status code used when signalling failure
   */
  this.statusCode =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedstatusCode == null) {
        that.cachedstatusCode = j_routingContext["statusCode()"]();
      }
      return that.cachedstatusCode;
    } else if (typeof __super_statusCode != 'undefined') {
      return __super_statusCode.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   If the route specifies produces matches, e.g. produces `text/html` and `text/plain`, and the `accept` header
   matches one or more of these then this returns the most acceptable match.

   @public

   @return {string} the most acceptable content type.
   */
  this.getAcceptableContentType =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["getAcceptableContentType()"]() ;
    } else if (typeof __super_getAcceptableContentType != 'undefined') {
      return __super_getAcceptableContentType.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The headers:
   <ol>
   <li>Accept</li>
   <li>Accept-Charset</li>
   <li>Accept-Encoding</li>
   <li>Accept-Language</li>
   <li>Content-Type</li>
   </ol>
   Parsed into {@link ParsedHeaderValue}

   @public

   @return {ParsedHeaderValues} A container with the parsed headers.
   */
  this.parsedHeaders =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedparsedHeaders == null) {
        that.cachedparsedHeaders = utils.convReturnVertxGen(ParsedHeaderValues, j_routingContext["parsedHeaders()"]());
      }
      return that.cachedparsedHeaders;
    } else if (typeof __super_parsedHeaders != 'undefined') {
      return __super_parsedHeaders.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a handler that will be called just before headers are written to the response. This gives you a hook where
   you can write any extra headers before the response has been written when it will be too late.

   @public
   @param handler {function} the handler 
   @return {number} the id of the handler. This can be used if you later want to remove the handler.
   */
  this.addHeadersEndHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return j_routingContext["addHeadersEndHandler(io.vertx.core.Handler)"](handler) ;
    } else if (typeof __super_addHeadersEndHandler != 'undefined') {
      return __super_addHeadersEndHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove a headers end handler

   @public
   @param handlerID {number} the id as returned from {@link RoutingContext#addHeadersEndHandler}. 
   @return {boolean} true if the handler existed and was removed, false otherwise
   */
  this.removeHeadersEndHandler =  function(handlerID) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_routingContext["removeHeadersEndHandler(int)"](handlerID) ;
    } else if (typeof __super_removeHeadersEndHandler != 'undefined') {
      return __super_removeHeadersEndHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Provides a handler that will be called after the last part of the body is written to the wire.
   The handler is called asynchronously of when the response has been received by the client.
   This provides a hook allowing you to do more operations once the request has been sent over the wire.
   Do not use this for resource cleanup as this handler might never get called (e.g. if the connection is reset).

   @public
   @param handler {function} the handler 
   @return {number} the id of the handler. This can be used if you later want to remove the handler.
   */
  this.addBodyEndHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return j_routingContext["addBodyEndHandler(io.vertx.core.Handler)"](handler) ;
    } else if (typeof __super_addBodyEndHandler != 'undefined') {
      return __super_addBodyEndHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Remove a body end handler

   @public
   @param handlerID {number} the id as returned from {@link RoutingContext#addBodyEndHandler}. 
   @return {boolean} true if the handler existed and was removed, false otherwise
   */
  this.removeBodyEndHandler =  function(handlerID) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return j_routingContext["removeBodyEndHandler(int)"](handlerID) ;
    } else if (typeof __super_removeBodyEndHandler != 'undefined') {
      return __super_removeBodyEndHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {boolean} true if the context is being routed to failure handlers.
   */
  this.failed =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["failed()"]() ;
    } else if (typeof __super_failed != 'undefined') {
      return __super_failed.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the body. Used by the {@link BodyHandler}. You will not normally call this method.

   @public
   @param body {Buffer} the body 
   */
  this.setBody =  function(body) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_routingContext["setBody(io.vertx.core.buffer.Buffer)"](body._jdel);
    } else if (typeof __super_setBody != 'undefined') {
      return __super_setBody.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the session. Used by the {@link SessionHandler}. You will not normally call this method.

   @public
   @param session {Session} the session 
   */
  this.setSession =  function(session) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_routingContext["setSession(io.vertx.ext.web.Session)"](session._jdel);
    } else if (typeof __super_setSession != 'undefined') {
      return __super_setSession.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the user. Usually used by auth handlers to inject a User. You will not normally call this method.

   @public
   @param user {User} the user 
   */
  this.setUser =  function(user) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      j_routingContext["setUser(io.vertx.ext.auth.User)"](user._jdel);
    } else if (typeof __super_setUser != 'undefined') {
      return __super_setUser.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Clear the current user object in the context. This usually is used for implementing a log out feature, since the
   current user is unbounded from the routing context.

   @public

   */
  this.clearUser =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_routingContext["clearUser()"]();
    } else if (typeof __super_clearUser != 'undefined') {
      return __super_clearUser.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the acceptable content type. Used by

   @public
   @param contentType {string} the content type 
   */
  this.setAcceptableContentType =  function(contentType) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'string' || __args[0] == null)) {
      j_routingContext["setAcceptableContentType(java.lang.String)"](contentType);
    } else if (typeof __super_setAcceptableContentType != 'undefined') {
      return __super_setAcceptableContentType.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Restarts the current router with a new method and path. All path parameters are then parsed and available on the
   params list. Query params will also be allowed and available.

   @public
   @param method {Object} the new http request 
   @param path {string} the new http path. 
   */
  this.reroute =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_routingContext["reroute(java.lang.String)"](__args[0]);
    } else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'string') {
      j_routingContext["reroute(io.vertx.core.http.HttpMethod,java.lang.String)"](io.vertx.core.http.HttpMethod.valueOf(__args[0]), __args[1]);
    } else if (typeof __super_reroute != 'undefined') {
      return __super_reroute.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns the languages for the current request. The languages are determined from the <code>Accept-Language</code>
   header and sorted on quality.

   When 2 or more entries have the same quality then the order used to return the best match is based on the lowest
   index on the original list. For example if a user has en-US and en-GB with same quality and this order the best
   match will be en-US because it was declared as first entry by the client.

   @public

   @return {Array.<LanguageHeader>} The best matched language for the request
   */
  this.acceptableLanguages =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedacceptableLanguages == null) {
        that.cachedacceptableLanguages = utils.convReturnListSetVertxGen(j_routingContext["acceptableLanguages()"](), LanguageHeader);
      }
      return that.cachedacceptableLanguages;
    } else if (typeof __super_acceptableLanguages != 'undefined') {
      return __super_acceptableLanguages.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Helper to return the user preferred language.
   It is the same action as returning the first element of the acceptable languages.

   @public

   @return {LanguageHeader} the users preferred locale.
   */
  this.preferredLanguage =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedpreferredLanguage == null) {
        that.cachedpreferredLanguage = utils.convReturnVertxGen(LanguageHeader, j_routingContext["preferredLanguage()"]());
      }
      return that.cachedpreferredLanguage;
    } else if (typeof __super_preferredLanguage != 'undefined') {
      return __super_preferredLanguage.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns a map of named parameters as defined in path declaration with their actual values

   @public

   @return {Object.<string, string>} the map of named parameters
   */
  this.pathParams =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnMapUnknown(j_routingContext["pathParams()"]()) ;
    } else if (typeof __super_pathParams != 'undefined') {
      return __super_pathParams.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the value of a single path parameter

   @public
   @param name {string} the name of parameter as defined in path declaration 
   @return {string} the actual value of the parameter or null if it doesn't exist
   */
  this.pathParam =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_routingContext["pathParam(java.lang.String)"](name) ;
    } else if (typeof __super_pathParam != 'undefined') {
      return __super_pathParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns a map of all query parameters inside the <a href="https://en.wikipedia.org/wiki/Query_string">query string</a><br/>
   The query parameters are lazily decoded: the decoding happens on the first time this method is called. If the query string is invalid
   it fails the context

   @public

   @return {MultiMap} the multimap of query parameters
   */
  this.queryParams =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(MultiMap, j_routingContext["queryParams()"]()) ;
    } else if (typeof __super_queryParams != 'undefined') {
      return __super_queryParams.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the value of a single query parameter. For more info {@link RoutingContext#queryParams}

   @public
   @param name {string} The name of query parameter 
   @return {Array.<string>} The list of all parameters matching the parameter name. It returns an empty list if no query parameter with <code>name</code> was found
   */
  this.queryParam =  function(name) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return j_routingContext["queryParam(java.lang.String)"](name) ;
    } else if (typeof __super_queryParam != 'undefined') {
      return __super_queryParam.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set Content-Disposition get to "attachment" with optional <code>filename</code> mime type.

   @public
   @param filename {string} the filename for the attachment 
   @return {RoutingContext}
   */
  this.attachment =  function(filename) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_routingContext["attachment(java.lang.String)"](filename) ;
      return that;
    } else if (typeof __super_attachment != 'undefined') {
      return __super_attachment.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Perform a 302 redirect to <code>url</code>. If a custom 3xx code is already defined, then that
   one will be preferred.
   <p/>
   The string "back" is special-cased
   to provide Referrer support, when Referrer
   is not present "/" is used.
   <p/>
   Examples:
   <p/>
   redirect('back');
   redirect('/login');
   redirect('http://google.com');

   @public
   @param url {string} the target url 
   @return {Future}
   */
  this.redirect =  function(url) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Future, j_routingContext["redirect(java.lang.String)"](url), undefined) ;
    } else if (typeof __super_redirect != 'undefined') {
      return __super_redirect.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Encode an Object to JSON and end the request.
   The method will apply the correct content type to the response,
   perform the encoding and end.

   @public
   @param json {Object} the json 
   @return {Future} a future to handle the end of the request
   */
  this.json =  function(json) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] !== 'function') {
      return utils.convReturnVertxGen(Future, j_routingContext["json(java.lang.Object)"](utils.convParamTypeUnknown(json)), undefined) ;
    } else if (typeof __super_json != 'undefined') {
      return __super_json.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Check if the incoming request contains the "Content-Type"
   get field, and it contains the give mime `type`.
   If there is no request body, `false` is returned.
   If there is no content type, `false` is returned.
   Otherwise, it returns true if the `type` that matches.
   <p/>
   Examples:
   <p/>
   // With Content-Type: text/html; charset=utf-8
   is("html"); // => true
   is("text/html"); // => true
   <p/>
   // When Content-Type is application/json
   is("application/json"); // => true
   is("html"); // => false

   @public
   @param type {string} content type 
   @return {boolean} The most close value
   */
  this.is =  function(type) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      if (that.cachedis == null) {
        that.cachedis = j_routingContext["is(java.lang.String)"](type);
      }
      return that.cachedis;
    } else if (typeof __super_is != 'undefined') {
      return __super_is.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Check if the request is fresh, aka
   Last-Modified and/or the ETag
   still match.

   @public

   @return {boolean} true if content is fresh according to the cache.
   */
  this.isFresh =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_routingContext["isFresh()"]() ;
    } else if (typeof __super_isFresh != 'undefined') {
      return __super_isFresh.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the ETag of a response.
   This will normalize the quotes if necessary.
   <p/>
   etag('md5hashsum');
   etag('"md5hashsum"');
   ('W/"123456789"');

   @public
   @param etag {string} the etag value 
   @return {RoutingContext}
   */
  this.etag =  function(etag) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_routingContext["etag(java.lang.String)"](etag) ;
      return that;
    } else if (typeof __super_etag != 'undefined') {
      return __super_etag.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the Last-Modified date using a String.

   @public
   @param instant {string} the last modified instant 
   @return {RoutingContext}
   */
  this.lastModified =  function(instant) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      j_routingContext["lastModified(java.lang.String)"](instant) ;
      return that;
    } else if (typeof __super_lastModified != 'undefined') {
      return __super_lastModified.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Shortcut to the response end.

   @public

   @return {Future} future
   */
  this.end =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Future, j_routingContext["end(java.lang.String)"](__args[0]), undefined) ;
    } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
      return utils.convReturnVertxGen(Future, j_routingContext["end(io.vertx.core.buffer.Buffer)"](__args[0]._jdel), undefined) ;
    } else if (__args.length === 0) {
      return utils.convReturnVertxGen(Future, j_routingContext["end()"](), undefined) ;
    } else if (typeof __super_end != 'undefined') {
      return __super_end.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_routingContext;
};

RoutingContext._jclass = utils.getJavaClass("io.vertx.ext.web.RoutingContext");
RoutingContext._jtype = {accept: function(obj) {
    return RoutingContext._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(RoutingContext.prototype, {});
    RoutingContext.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
RoutingContext._create = function(jdel) {var obj = Object.create(RoutingContext.prototype, {});
  RoutingContext.apply(obj, arguments);
  return obj;
}
module.exports = RoutingContext;
var Promise = require('vertx-js/promise');
