package io.vertx.ext.mongo;

import io.vertx.codegen.annotations.VertxGen;
import io.vertx.core.Future;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonObject;
import io.vertx.core.streams.ReadStream;
import io.vertx.core.streams.WriteStream;

import java.util.List;

@VertxGen
public interface MongoGridFsClient {
  /**
   * Deletes a file by it's ID
   *
   * @param id            the identifier of the file
   * @return a future notified when the operation completes
   */
  Future<Void> delete(String id);

  /**
   * Read file by name to ReadStream
   */
  ReadStream<Buffer> readByFileName(String fileName);

  /**
   * Read file by name to ReadStream with options
   */
  ReadStream<Buffer> readByFileNameWithOptions(String fileName, GridFsDownloadOptions options);

  /**
   * Read file by id to ReadStream
   */
  ReadStream<Buffer> readById(String id);

  Future<Long> downloadByFileName(WriteStream<Buffer> stream, String fileName);

  Future<Long> downloadByFileNameWithOptions(WriteStream<Buffer> stream, String fileName, GridFsDownloadOptions options);

  Future<Long> downloadById(WriteStream<Buffer> stream, String id);

  /**
   * Downloads a file.
   *
   * @param fileName      the name of the file to download
   * @return a future notified with the result
   */
  Future<Long> downloadFile(String fileName);

  /**
   * Downloads a file and gives it a new name.
   *
   * @param fileName      the name of the file to download
   * @param newFileName   the name the file should be saved as
   * @return a future notified with the result
   */
  Future<Long> downloadFileAs(String fileName, String newFileName);

  /**
   * Downloads a file using the ID generated by GridFs.
   *
   * @param id            the GridFs Object ID of the file to download
   * @return a future notified with the result
   */
  Future<Long> downloadFileByID(String id, String fileName);

  /**
   * Drops the entire file bucket with all of its contents
   *
   * @return a future notified when the operation completes
   */
  Future<Void> drop();

  /**
   * Finds all file ids in the bucket
   *
   * @return a future notified with the result
   */
  Future<List<String>> findAllIds();

  /**
   * Finds all file ids that match a query.
   *
   * @param query         a bson query expressed as json that will be used to match files
   * @return a future notified with the result
   */
  Future<List<String>> findIds(JsonObject query);

  Future<String> uploadByFileName(ReadStream<Buffer> stream, String fileName);

  Future<String> uploadByFileNameWithOptions(ReadStream<Buffer> stream, String fileName, GridFsUploadOptions options);

  /**
   * Upload a file to gridfs
   *
   * @param fileName      the name of the file to store in gridfs
   * @return a future notified with the result
   */
  Future<String> uploadFile(String fileName);

  /**
   * Upload a file to gridfs with options
   *
   * @param fileName      the name of the file to store in gridfs
   * @param options       {@link GridFsUploadOptions} for specifying metadata and chunk size
   * @return a future notified with the result
   */
  Future<String> uploadFileWithOptions(String fileName, GridFsUploadOptions options);

  /**
   * Close the client and release its resources
   */
  void close();
}
